﻿using FluentValidation;
using KarleyLibrary.Erweiterungen;
using System;
using System.Linq;
using WK5.Core.Models;
using WK5.Core.PageModels.Verkauf;

namespace WK5.Core.Validators.Verkauf.Angebote
{
    public class AngebotValidator : AbstractValidator<Angebot>
    {

        public AngebotValidator(IValidator<Belegposition> posValidator)
        {
            RuleFor(angebot => angebot.Name1)
                .NotEmpty()
                .MaximumLength(60);

            RuleFor(angebot => angebot.Name2)
                .MaximumLength(60);

            RuleFor(angebot => angebot.Name3)
                .MaximumLength(60);

            RuleFor(angebot => angebot.Anrede)
                .MaximumLength(40)
                .ISO88591();

            RuleFor(angebot => angebot.Ansprechpartner)
                .MaximumLength(40)
                .ISO88591();

            RuleFor(angebot => angebot.Strasse)
                .NotEmpty()
                .MaximumLength(40);

            RuleFor(angebot => angebot.Land)
                .NotEmpty()
                .MaximumLength(2)
                .ISO88591();

            RuleFor(angebot => angebot.Postleitzahl)
                .NotEmpty()
                .MaximumLength(8);

            RuleFor(angebot => angebot.Ort)
                .NotEmpty()
                .MaximumLength(40);

            RuleFor(x => x.Notiz)
                .Must((notiz) => HasNoRtf(notiz))
                .WithMessage("Notiz ist noch im alten Format. Bitte prüfen und korrigieren.");

            RuleFor(angebot => angebot.Frachtkosten)
                .GreaterThanOrEqualTo(0);

            RuleFor(x => x.LieferbedingungId)
                .GreaterThan(0);

            RuleFor(x => x.ZahlungsbedingungId)
                .GreaterThan(0);

            RuleFor(angebot => angebot.Verpackungskosten)
                .GreaterThanOrEqualTo(0);

            RuleFor(angebot => angebot.Versicherungskosten)
                .GreaterThanOrEqualTo(0);            

            RuleFor(x => x.Kopftext)
                .Must((kopftext) => HasNoRtf(kopftext))
                .WithMessage("Kopftext ist noch im alten Format. Bitte prüfen und korrigieren.");            

            RuleFor(x => x.Fußtext)
                .Must((fußtext) => HasNoRtf(fußtext))
                .WithMessage("Fußtext ist noch im alten Format. Bitte prüfen und korrigieren.");



            RuleFor(angebot => angebot.Positionen)
                .Must((positionen) => positionen.Any())
                .WithMessage("Das Angebot muss mindestens über eine Position verfügen.");


            RuleFor(angebot => angebot.AnfrageDurch)
              .MaximumLength(30)
              .NotNull()
              .NotEmpty()
              .When(x => x.Belegnummer is 0);

            RuleFor(angebot => angebot.AnfrageDurch)
              .ISO88591();

            RuleFor(angebot => angebot.Anfragennummer)
               .MaximumLength(100)
               .NotNull()
               .NotEmpty()
               .When(x => x.Belegnummer is 0);

            RuleFor(x => x.Anfragennummer)
                .ISO88591();

            RuleForEach(angebot => angebot.Positionen)
                .SetValidator(posValidator);

            RuleFor(x => x.Lieferzeit)
                .Must((lieferzeit) => HasNoRtf(lieferzeit))
                .WithMessage("Lieferzeit ist noch im alten Format. Bitte prüfen und korrigieren.");

            RuleFor(x => x.Ablehnungsgrund)
                .GreaterThan(0)
                .WithMessage("Bitte wählen Sie einen Ablehnungsgrund aus.")
                .When(x => x.BELE_L_ABGELEHNT);

            RuleFor(x => x.Liefertermin)
                .Must((termin) => termin == default(DateTime) ? true : termin >= DateTime.Now.Date)                
                .WithMessage("Der Liefertermin darf nicht in der Vergangenheit liegen")
                .When(x => x.Belegnummer is 0);
        }

        private bool HasNoRtf(string text)
        {
            return !text.StartsWith(@"{\rtf1");
        }
    }
}
