﻿using iText.IO.Font;
using iText.Kernel.Font;
using iText.Kernel.Pdf;
using iText.Kernel.Pdf.Action;
using iText.Kernel.Pdf.Filespec;
using iText.Layout;
using iText.Layout.Element;
using iText.Pdfa;
using KarleyLibrary.Erweiterungen;
using System;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using WK5.Core.Models;
using WK5.Core.Models.VatChecker;

namespace WK5.Core.Services
{
    public class VatService : IDisposable
    {
        private readonly HttpClient _httpClient;
        public const string VAT_URL = "https://evatr.bff-online.de/evatrRPC?UstId_1=DE251382615&UstId_2=";
        public const int GÜLTIGKEITSDAUER_IN_TAGEN = 1; // 04.07.2022 MK: Wir prüfen absofort jeden Tag.

        private static List<int> GültigeResponseCodes = new List<int>
        {
            200,
            223
        };

        public VatService(HttpClient httpClient)
        {
            _httpClient = httpClient;
            httpClient.Timeout = TimeSpan.FromSeconds(5);
            httpClient.DefaultRequestHeaders.Add("Accept", "test/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8");

        }
        public async Task<VatCheck> CheckVatAsync(VatRequest request, FbController2 fbController, bool pdfaGenerieren = true)
        {
            // Erstmal prüfen, ob wir überhaupt Anfragen müssen
            var (PrüfungIstNotwendig, ZuletztGeprüft) = await IstPrüfungNotwendigAsync(request, fbController);
            if (!request.Drucken && !PrüfungIstNotwendig)
            {
                return new VatCheck
                {
                    Datum = DateTime.Now,
                    Druck = false,
                    Erg_Name = "A",
                    Erg_Ort = "A",
                    Erg_PLZ = "A",
                    Erg_Str = "A",
                    ErrorCode = 200,
                    Firmenname = request.Firmenname,
                    Ort = request.Ort,
                    PLZ = request.Postleitzahl,
                    Strasse = request.Straße,
                    Uhrzeit = DateTime.Now,
                    Message = $"Die letzte erfolgreiche Prüfung war am {ZuletztGeprüft.ToShortDateString()}. Eine weitere Prüfung ist erst ab dem {ZuletztGeprüft.AddDays(GÜLTIGKEITSDAUER_IN_TAGEN).ToShortDateString()} notwendig",
                    PrüfungNichtNotwendig = true
                };
            }

            //Wenn das Land der UstId DE ist macht es keinen Sinn zu prüfen ob die VatId stimmt
            if (request.UstId.StartsWith("DE"))
            {
                return new VatCheck
                {
                    Erg_Name = "A",
                    Erg_Ort = "A",
                    Erg_PLZ = "A",
                    Erg_Str = "A",
                    ErrorCode = 200,
                    Firmenname = request.Firmenname,
                    Ort = request.Ort,
                    PLZ = request.Postleitzahl,
                    Strasse = request.Straße,
                    Gueltig_ab = DateTime.Now.AddYears(-5),
                    Gueltig_bis = DateTime.Now.AddYears(5),
                    Datum = DateTime.Now,
                    Druck = true,
                    Uhrzeit = DateTime.Now,
                    Message = "Ist eine DE Nummer also immer erfolgreich! #TAX_LIB_1"
                };
            }

            await foreach (var land in Land.GetLänderAsync(fbController).Where(x => !x.WK5_LAND_L_ISTEULAND))
            {
                if (request.UstId.StartsWith(land.LAND_A_ID))
                {
                    return new VatCheck
                    {
                        Erg_Name = "B",
                        Erg_Ort = "B",
                        Erg_PLZ = "B",
                        Erg_Str = "B",
                        ErrorCode = 212,
                        Firmenname = request.Firmenname,
                        Ort = request.Ort,
                        PLZ = request.Postleitzahl,
                        Strasse = request.Straße,
                        Gueltig_ab = new DateTime(1990, 1, 1, 1, 1, 1, 1),
                        Gueltig_bis = new DateTime(1990, 1, 1, 1, 1, 1, 1),
                        Datum = DateTime.Now,
                        Druck = false,
                        Uhrzeit = DateTime.Now,
                        Message = "Ist eine nicht EU Nummer, also immer negativ! #TAX_LIB_2"
                    };
                }
            }
            Uri uri = new Uri(VAT_URL + request.GetParameter());
            HttpResponseMessage response = await SendRequestAsync(uri);

            List<string> tokens = new List<string>
            {
                "ErrorCode",
                "UstId_1",
                "UstId_2",
                "Druck",
                "Erg_PLZ",
                "Erg_Ort",
                "Erg_Str",
                "Erg_Name",
                "Ort",
                "Datum",
                "PLZ",
                "Uhrzeit",
                "Gueltig_ab",
                "Gueltig_bis",
                "Strasse",
                "Firmenname"
            };
            List<string> removeTokens = new List<string>
            {
                "<value>",
                "</value>",
                "<data>",
                "</data>",
                "<array>",
                "</array>",
                "</string>",
                "<param>",
                "</param>",
                "<params>",
                "</params>"
            };

            string content = await response.Content.ReadAsStringAsync();


            List<string> xmlElems = content.Split(new string[] { "<param>" }, StringSplitOptions.RemoveEmptyEntries).ToList();
            Dictionary<string, string> results = new Dictionary<string, string>();


            for (int i = 0; i < xmlElems.Count; i++)
            {
                if (!xmlElems[i].Contains("<string>"))
                {
                    continue;
                }

                xmlElems[i] = xmlElems[i].Replace("\r", String.Empty);
                xmlElems[i] = xmlElems[i].Replace("\n", String.Empty);
                foreach (string rm in removeTokens)
                {
                    xmlElems[i] = xmlElems[i].Replace(rm, String.Empty);
                }
                string[] keyVal = xmlElems[i].Split(new string[] { "<string>" }, StringSplitOptions.RemoveEmptyEntries);

                if (keyVal.Length == 1)
                {
                    if (tokens.Contains(keyVal[0]))
                    {
                        results.Add(keyVal[0], "");
                    }
                    else
                    {
                        results.Add("", keyVal[0]);
                    }
                }
                else if (keyVal.Length == 2)
                {
                    results.Add(keyVal[0] ?? "", keyVal[1] ?? "");
                }


            }

            VatCheck vatCheck = DictToVatCheck(results);
            vatCheck.Message = GetMessage(vatCheck.ErrorCode);
            bool istGültig = IsVatCheckValid(vatCheck);

            await UpdateZuordnungAsync(request, istGültig, fbController);


            if(istGültig && pdfaGenerieren)
            {
                string path = await GlobalConfig.GetConfigAsync(GlobalConfig.SCANINPUT_PFAD_CONFIG_NAME, fbController) as string ?? "";
                if (!Directory.Exists(path) || path.Length == 0)
                {
                    throw new DirectoryNotFoundException($"Der Pfad \"{path}\" konnte nicht gefunden werden.");
                }

                await PdfaGenerierenAsync(Path.Combine(path, $"VatResponse_{request.UstId}.pdf"), content, request.Kundennummer, request.UstId, uri.ToString(), vatCheck);
            }

            return vatCheck;
        }

        /// <summary>
        /// Stellt fest, ob für eine Prüfung für die VatRequest durchgeführt werden muss, oder nicht.
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        private async Task<(bool PrüfungIstNotwendig, DateTime ZuletztGeprüft)> IstPrüfungNotwendigAsync(VatRequest request, FbController2 fbController)
        {
            DataRow? row;
            bool bereitsGültig;
            DateTime zuletztGeprüft;

            if (request.Zuordnung is VatZuordnung.Kunde)
            {
                fbController.AddParameter("@KUND_A_NR", request.Kundennummer);
                row = await fbController.SelectRowAsync("SELECT KUND_WK5_D_USTID_GEPRUEFT_DATE, KUND_WK5_L_USTID_GUELTIG FROM KUNDEN WHERE KUND_A_NR = @KUND_A_NR");
                
                // Kunde nicht gefunden
                if (row is null)
                {
                    throw new ArgumentException($"Es konnte kein Kunde mit der Kundennummer {request.Kundennummer} gefunden werden");
                }

                bereitsGültig = row["KUND_WK5_L_USTID_GUELTIG"] as string == "Y";
                zuletztGeprüft = row.Field<DateTime?>("KUND_WK5_D_USTID_GEPRUEFT_DATE") ?? default;
            }
            else if(request.Zuordnung is VatZuordnung.Lieferant)
            {
                return (true, DateTime.Now.AddDays(-GÜLTIGKEITSDAUER_IN_TAGEN));
            }
            else
            {
                fbController.AddParameter("@KULA_A_KUNDNR", request.Kundennummer);
                fbController.AddParameter("@KULA_N_NR", request.LieferanschriftId);
                row = await fbController.SelectRowAsync("SELECT KULA_WK5_D_USTID_GEPRUEFT_DATE, KULA_WK5_L_USTID_GUELTIG FROM LIEFERANSCHRIFTEN WHERE KULA_A_KUNDNR = @KULA_A_KUNDNR AND KULA_N_NR = @KULA_N_NR");

                // Lieferanschrift nicht gefunden
                if (row is null)
                {
                    throw new ArgumentException($"Es konnte keine Lieferanschrift {request.LieferanschriftId} für die Kundennummer {request.Kundennummer} gefunden werden");
                }

                bereitsGültig = row["KULA_WK5_L_USTID_GUELTIG"] as string == "Y";
                zuletztGeprüft = row.Field<DateTime?>("KULA_WK5_D_USTID_GEPRUEFT_DATE") ?? default;
            }

            bool prüfungIstnotwendig = !((DateTime.Now - zuletztGeprüft).TotalDays < GÜLTIGKEITSDAUER_IN_TAGEN && bereitsGültig);

            return (prüfungIstnotwendig, zuletztGeprüft);
        }
        public async Task UpdateZuordnungAsync(VatRequest request, bool istGültig, FbController2 fbController)
        {
            if (request.Zuordnung is VatZuordnung.Kunde)
            {
                fbController.AddParameter("@KUND_WK5_L_USTID_GUELTIG", istGültig);
                fbController.AddParameter("@KUND_L_MWST", !istGültig);
                fbController.AddParameter("@KUND_A_NR", request.Kundennummer);
                await fbController.QueryAsync("UPDATE KUNDEN SET KUND_L_MWST = @KUND_L_MWST, KUND_WK5_L_USTID_GUELTIG = @KUND_WK5_L_USTID_GUELTIG, KUND_WK5_D_USTID_GEPRUEFT_DATE = CURRENT_TIMESTAMP WHERE KUND_A_NR = @KUND_A_NR");

                fbController.AddParameter("@KULA_WK5_L_USTID_GUELTIG", istGültig);
                //fbController.AddParameter("@KULA_A_LAND", request.UstId[..2]); // Die ersten zwei zeichen sind das land
                fbController.AddParameter("@KULA_L_MWST", !istGültig);
                fbController.AddParameter("@KULA_A_KUNDNR", request.Kundennummer);
                await fbController.QueryAsync("UPDATE LIEFERANSCHRIFTEN SET KULA_L_MWST = @KULA_L_MWST, KULA_WK5_L_USTID_GUELTIG = @KULA_WK5_L_USTID_GUELTIG, KULA_WK5_D_USTID_GEPRUEFT_DATE = CURRENT_TIMESTAMP WHERE KULA_A_LAND = (SELECT KUND_A_LAND FROM KUNDEN WHERE KUND_A_NR = @KULA_A_KUNDNR) AND KULA_A_KUNDNR = @KULA_A_KUNDNR");
            }
            else if(request.Zuordnung is VatZuordnung.Lieferanschrift)
            {
                fbController.AddParameter("@KULA_WK5_L_USTID_GUELTIG", istGültig);
                fbController.AddParameter("@KULA_N_NR", request.LieferanschriftId);
                fbController.AddParameter("@KULA_L_MWST", !istGültig);
                fbController.AddParameter("@KULA_A_KUNDNR", request.Kundennummer);
                await fbController.QueryAsync("UPDATE LIEFERANSCHRIFTEN SET KULA_L_MWST = @KULA_L_MWST, KULA_WK5_L_USTID_GUELTIG = @KULA_WK5_L_USTID_GUELTIG, KULA_WK5_D_USTID_GEPRUEFT_DATE = CURRENT_TIMESTAMP WHERE KULA_N_NR = @KULA_N_NR AND KULA_A_KUNDNR = @KULA_A_KUNDNR");

                fbController.AddParameter("@KUND_WK5_L_USTID_GUELTIG", istGültig);
                fbController.AddParameter("@KUND_L_MWST", !istGültig);
                fbController.AddParameter("@KUND_A_USTIDNR", request.UstId);
                await fbController.QueryAsync("UPDATE KUNDEN SET KUND_L_MWST = @KUND_L_MWST, KUND_WK5_L_USTID_GUELTIG = @KUND_WK5_L_USTID_GUELTIG, KUND_WK5_D_USTID_GEPRUEFT_DATE = CURRENT_TIMESTAMP WHERE KUND_A_USTIDNR = @KUND_A_USTIDNR");
            }
        }
        private async Task<HttpResponseMessage> SendRequestAsync(Uri uri)
        {
            try
            {
                HttpResponseMessage response = await _httpClient.GetAsync(uri);
                if (response.IsSuccessStatusCode)
                {
                    return response;
                }
                else
                {
                    throw new WebException("Der Vat-Check Dienst ist momentan nicht erreichbar");
                }
            }
            catch (Exception ex)
            {
                throw new WebException("Der Vat-Check Dienst ist momentan nicht erreichbar", ex);
            }
        }

        /// <summary>
        /// Generiert das Antwort-PDF, welches wir in ecoDms archivieren.
        /// </summary>
        /// <param name="pfad"></param>
        /// <param name="text"></param>
        /// <param name="kundennummer"></param>
        /// <param name="ustidnr"></param>
        /// <param name="anfrageUrl"></param>
        /// <param name="result"></param>
        private Task PdfaGenerierenAsync(string pfad, string text, string kundennummer, string ustidnr, string anfrageUrl, VatCheck result)
        {
            string fontPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "ressourcen/arial.ttf");

            string icmPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "ressourcen/sRGB_IEC61966-2-1_no_black_scaling.icc");

            string path = StringErweiterung.GetUniqueFileName(pfad);

            PdfWriter pdfWriter = new PdfWriter(path);
            PdfADocument pdfDoc = new PdfADocument(pdfWriter, PdfAConformanceLevel.PDF_A_3A, new PdfOutputIntent("Custom", "", "http://www.color.org", "sRGB IEC61966-2.1", new FileStream(icmPath, FileMode.Open, FileAccess.Read)));

            Document doc = new Document(pdfDoc);

            pdfDoc.SetTagged();
            pdfDoc.GetCatalog().SetLang(new PdfString("de-DE"));
            pdfDoc.GetCatalog().SetViewerPreferences(new PdfViewerPreferences().SetDisplayDocTitle(true));
            PdfDocumentInfo pdfDocInfo = pdfDoc.GetDocumentInfo();
            pdfDocInfo.SetTitle("Antwort auf die UstID Anfrage für KD-" + kundennummer);

            PdfFont pdfFont = PdfFontFactory.CreateFont(fontPath, PdfEncodings.WINANSI, PdfFontFactory.EmbeddingStrategy.FORCE_EMBEDDED);
            Paragraph title = new Paragraph();
            title.SetFont(pdfFont);
            title.Add(new Text($"UstId Validierung über die XML-RPC Schnittstelle des BZSt.\r\n\r\n").SetFontSize(12.0f).SetBold());
            title.Add(new Text($"Datum der Ausführung: {DateTime.Now:dd.MM.yyyy - HH:mm:ss}\r\n"));
            title.Add(new Text($"Angefragte UstID: {ustidnr}\r\n"));
            title.Add(new Text($"zur Kundennummer: {kundennummer}\r\n"));
            title.Add(new Text($"Programm / Version: TaxLibrary / 2.0.0.0\r\n\r\n"));
            title.Add(new Text($"Ergebnis der Prüfung:").SetBold());

            //title.Add(new Text($"Empfangener Datensatz der XML-RPC Schnittstelle des BzSt folgend und im Anhang.\r\n\r\n"));
            doc.Add(title);

            Table tbl = new Table(9);
            tbl.SetFont(pdfFont);
            tbl.AddHeaderCell("USt-IdNr");
            tbl.AddHeaderCell("Datum - Uhrzeit");
            tbl.AddHeaderCell("Rückmeldung");
            tbl.AddHeaderCell("Name");
            tbl.AddHeaderCell("Ort");
            tbl.AddHeaderCell("PLZ");
            tbl.AddHeaderCell("Straße");
            tbl.AddHeaderCell("Code");
            tbl.AddHeaderCell("Nachricht");

            tbl.AddCell(ustidnr);
            tbl.AddCell(result.Datum.ToString("dd.MM.yyyy") + " - " + result.Uhrzeit.ToString("HH:mm:ss"));
            tbl.AddCell(GültigeResponseCodes.Contains(result.ErrorCode) ? "gültig" : "ungültig");
            tbl.AddCell(result.Erg_Name);
            tbl.AddCell(result.Erg_Ort);
            tbl.AddCell(result.Erg_PLZ);
            tbl.AddCell(result.Erg_Str);
            tbl.AddCell(result.ErrorCode.ToString());
            tbl.AddCell(new Cell().Add(new Paragraph(result.Message).SetFontSize(6.0f)));

            doc.Add(tbl);

            Paragraph erklärung = new Paragraph();
            erklärung.SetFont(pdfFont);
            erklärung.Add(new Text("A: Stimmt überein; ").SetFontSize(6.0f));
            erklärung.Add(new Text("B: Stimmt nicht überein; ").SetFontSize(6.0f));
            erklärung.Add(new Text("C: von Ihnen nicht angefragt; ").SetFontSize(6.0f));
            erklärung.Add(new Text("D: vom Mitgliedsstaat nicht mitgeteilt\r\n\r\n").SetFontSize(6.0f));
            erklärung.Add(new Link("Abfrage-URL\r\n", PdfAction.CreateURI(anfrageUrl)));
            doc.Add(erklärung);

            Paragraph xmlText = new Paragraph();
            xmlText.SetFont(pdfFont);
            xmlText.SetFont(pdfFont);
            xmlText.Add(new Text(text).SetFontSize(6.0f));
            doc.Add(xmlText);


            pdfDoc.AddFileAttachment("VatCheck Response", PdfFileSpec.CreateEmbeddedFileSpec(pdfDoc, new ASCIIEncoding().GetBytes(text), "VatCheck Response", "vat-response.xml", PdfName.ApplicationXml, new PdfDictionary(), PdfName.Alternative));

            doc.Close();

            return Task.CompletedTask;
        }

        #region Helper
        public bool IsVatCheckValid(VatCheck check)
        {
            bool name = check.Erg_Name.ToUpper() is "A" or "D";
            bool ort = check.Erg_Ort.ToUpper() is "A" or "D";
            bool plz = check.Erg_PLZ.ToUpper() is "A" or "D";
            bool str = check.Erg_Str.ToUpper() is "A" or "D";

            return GültigeResponseCodes.Contains(check.ErrorCode) && name && ort && plz && str;
        }

        private VatCheck DictToVatCheck(Dictionary<string, string> dict)
        {
            VatCheck tmp = new VatCheck();

            foreach (KeyValuePair<string, string> kvp in dict)
            {
                object? value;
                try
                {
                    value = tmp.GetType().GetProperty(kvp.Key);
                }
                catch
                {
                    continue;
                }

                if (value == null)
                {
                    continue;
                }
                else
                {
                    var prop = tmp.GetType().GetProperty(kvp.Key);
                    if (prop is not null)
                    {
                        if (prop.PropertyType == typeof(Int32))
                        {
                            bool success = Int32.TryParse(kvp.Value, out int val);
                            if (success)
                            {
                                prop.SetValue(tmp, val, null);
                            }
                        }
                        else if (prop.PropertyType == typeof(DateTime))
                        {
                            bool success = DateTime.TryParse(kvp.Value, out DateTime dt);
                            if (success)
                            {
                                prop.SetValue(tmp, dt, null);
                            }
                        }
                        else if (prop.Name == "Druck")
                        {
                            prop.SetValue(tmp, kvp.Value != "nein", null);
                        }
                        else
                        {
                            prop.SetValue(tmp, kvp.Value, null);
                        }
                    }

                }
            }

            return tmp;
        }

        private string GetMessage(int code)
        {
            return code switch
            {
                200 => "Die angefragte USt-IdNr. ist gültig.",
                201 => "Die angefragte USt-IdNr. ist ungültig.",
                202 => "Die angefragte USt-IdNr. ist ungültig. Sie ist nicht in der Unternehmerdatei des betreffenden EU-Mitgliedstaates registriert. Hinweis: Ihr Geschäftspartner kann seine gültige USt-IdNr. bei der für ihn zuständigen Finanzbehörde in Erfahrung bringen. Möglicherweise muss er einen Antrag stellen, damit seine USt-IdNr. in die Datenbank aufgenommen wird.",
                203 => "Die angefragte USt-IdNr. ist ungültig. Sie ist erst ab dem ... gültig (siehe Feld 'Gueltig_ab').",
                204 => "Die angefragte USt-IdNr. ist ungültig. Sie war im Zeitraum von ... bis ... gültig (siehe Feld 'Gueltig_ab' und 'Gueltig_bis').",
                205 => "Ihre Anfrage kann derzeit durch den angefragten EU-Mitgliedstaat oder aus anderen Gründen nicht beantwortet werden. Bitte versuchen Sie es später noch einmal. Bei wiederholten Problemen wenden Sie sich bitte an das Bundeszentralamt für Steuern - Dienstsitz Saarlouis.",
                206 => "Ihre deutsche USt-IdNr. ist ungültig. Eine Bestätigungsanfrage ist daher nicht möglich. Den Grund hierfür können Sie beim Bundeszentralamt für Steuern - Dienstsitz Saarlouis - erfragen.",
                207 => "Ihnen wurde die deutsche USt-IdNr. ausschliesslich zu Zwecken der Besteuerung des innergemeinschaftlichen Erwerbs erteilt. Sie sind somit nicht berechtigt, Bestätigungsanfragen zu stellen.",
                208 => "Für die von Ihnen angefragte USt-IdNr. läuft gerade eine Anfrage von einem anderen Nutzer. Eine Bearbeitung ist daher nicht möglich. Bitte versuchen Sie es später noch einmal.",
                209 => "Die angefragte USt-IdNr. ist ungültig. Sie entspricht nicht dem Aufbau der für diesen EU-Mitgliedstaat gilt. ( Aufbau der USt-IdNr. aller EU-Länder)",
                210 => "Die angefragte USt-IdNr. ist ungültig. Sie entspricht nicht den Prüfziffernregeln die für diesen EU-Mitgliedstaat gelten.",
                211 => "Die angefragte USt-IdNr. ist ungültig. Sie enthält unzulässige Zeichen (wie z.B. Leerzeichen oder Punkt oder Bindestrich usw.).",
                212 => "Die angefragte USt-IdNr. ist ungültig. Sie enthält ein unzulässiges Länderkennzeichen.",
                213 => "Die Abfrage einer deutschen USt-IdNr. ist nicht möglich.",
                214 => "Ihre deutsche USt-IdNr. ist fehlerhaft. Sie beginnt mit 'DE' gefolgt von 9 Ziffern.",
                215 => "Ihre Anfrage enthält nicht alle notwendigen Angaben für eine einfache Bestätigungsanfrage (Ihre deutsche USt-IdNr. und die ausl. USt-IdNr.). Ihre Anfrage kann deshalb nicht bearbeitet werden.",
                216 => "Ihre Anfrage enthält nicht alle notwendigen Angaben für eine qualifizierte Bestätigungsanfrage (Ihre deutsche USt-IdNr., die ausl. USt-IdNr., Firmenname einschl. Rechtsform und Ort). Es wurde eine einfache Bestätigungsanfrage durchgeführt mit folgenden Ergebnis: Die angefragte USt-IdNr. ist gültig.",
                217 => "Bei der Verarbeitung der Daten aus dem angefragten EU-Mitgliedstaat ist ein Fehler aufgetreten. Ihre Anfrage kann deshalb nicht bearbeitet werden.",
                218 => "Eine qualifizierte Bestätigung ist zur Zeit nicht möglich. Es wurde eine einfache Bestätigungsanfrage mit folgendem Ergebnis durchgeführt: Die angefragte USt-IdNr. ist gültig.",
                219 => "Bei der Durchführung der qualifizierten Bestätigungsanfrage ist ein Fehler aufgetreten. Es wurde eine einfache Bestätigungsanfrage mit folgendem Ergebnis durchgeführt:Die angefragte USt-IdNr. ist gültig.",
                220 => "Bei der Anforderung der amtlichen Bestätigungsmitteilung ist ein Fehler aufgetreten. Sie werden kein Schreiben erhalten.",
                221 => "Die Anfragedaten enthalten nicht alle notwendigen Parameter oder einen ungültigen Datentyp. Weitere Informationen erhalten Sie bei den Hinweisen zum Schnittstelle - Aufruf.",
                223 => "Die angefragte USt-IdNr. ist gültig. Die Druckfunktion steht nicht mehr zur Verfügung, da der Nachweis gem. UStAE zu § 18e.1 zu führen ist.",
                999 => "Eine Bearbeitung Ihrer Anfrage ist zurzeit nicht möglich. Bitte versuchen Sie es später noch einmal.",
                _ => "Der Code konnte nicht ausgewertet werden!",
            };
        }

        public void Dispose()
        {
            _httpClient.Dispose();
        }
        #endregion
    }
}
