﻿using ecoDMS.Library;
using Serilog;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace WK5.Core.Services
{
    /// <summary>
    /// Stellt Funktionen zur Vorschau von Dokumenten aus ecoDMS in Blazor zur Verfügung.
    /// </summary>
    public class EcoDmsVorschauService
    {
        private List<string> _ecoVorschauBilder = new List<string>();
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob gerade ein Dokument aus ecoDMS geladen wird, oder nicht.
        /// </summary>
        public bool LoadingEcoDokument { get; private set; }
        public bool ShowModal { get; set; }
        public string? SelectedBelegnummer { get; private set; }


        /// <summary>
        /// Lädt die Vorschaubilder eines Dokuments aus ecoDMS
        /// </summary>
        /// <param name="belegnummer"></param>
        /// <param name="istRechnung">Wenn true, dann muss die Dokumentenart Rechnungseingang sein.</param>
        /// <returns></returns>
        public async Task GetEcoDokumentAsync(string belegnummer, bool istRechnung = true)
        {
            ShowModal = true;
            LoadingEcoDokument = true;
            _ecoVorschauBilder.Clear();
            SelectedBelegnummer = belegnummer;
            EcoDmsController ecoDmsController = await EcoDmsController.CreateAsync(new Uri(GlobalConfig.ECODMS_BASIS_URL), GlobalConfig.ECODMS_DEFAULT_USER, Log.Logger);

            var suchFilter = new List<EcoSearchFilter>
            {
                new EcoSearchFilter
                {
                    ClassifyAttribut = ecoDmsController.ClassifyAttributes.GetKeyByValue("Belegnummer")!,
                    SearchOperator = SuchOperator.Gleich,
                    SearchValue = belegnummer.Trim()
                }
            };

            if (istRechnung)
            {
                suchFilter.Add(new EcoSearchFilter
                {
                    ClassifyAttribut = "docart",
                    SearchOperator = SuchOperator.Gleich,
                    SearchValue = "15" // Rechnung
                });
            }
            else
            {
                suchFilter.Add(new EcoSearchFilter
                {
                    ClassifyAttribut = "docart",
                    SearchOperator = SuchOperator.Gleich,
                    SearchValue = "11" // Lieferschein
                });
            }

            var sucheDokument = (await ecoDmsController.EcoSucheDoksAsync(suchFilter)).FirstOrDefault();


            if (sucheDokument is not null)
            {
                try
                {
                    int zähler = 1;
                    while (true)
                    {
                        var data = await ecoDmsController.GetDokumentenVorschauAsync(sucheDokument.DocId, zähler, 820);
                        string ecoBaseImage = Convert.ToBase64String(data);
                        _ecoVorschauBilder.Add(ecoBaseImage);
                        zähler++;
                    }
                }
                catch (Exception)
                {
                    // Wir laden solange, bis es zu einem Fehler kommt, so umgehen wir die Limits der ecoDMS-API
                }
            }

            LoadingEcoDokument = false;
        }

        /// <summary>
        /// Leert die Vorschaubilder.
        /// </summary>
        public void Clear()
        {
            SelectedBelegnummer = null;
            _ecoVorschauBilder.Clear();
            ShowModal = false;
        }
        public IEnumerable<string> GetVorschauBilder()
        {
            return _ecoVorschauBilder.AsEnumerable();
        }
        
    }
}
