﻿#if DEBUG
//#define keepAuftrag // Sorgt dafür, dass die Buchungen und die Freigabe eines AUftrags nicht gelöscht werden
//#define noTransaction // Zum debuggen um Änderungen schnell nachvollziehen zu können
#endif
using KarleyLibrary.Attributes;
using KarleyLibrary.Erweiterungen;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using WK5.Core.Services;

namespace WK5.Core.Models.Tools.Lagerregal
{
    public sealed class PackAuftrag : PackBeleg
    {
        #region Eigenschaften
        public int Id { get; set; } = 1;
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob der Beleg zum Packen freigegeben ist, oder nicht.
        /// </summary>
        public bool ZumPackenFreigegeben => !BELE_L_ERLEDIGT && !BELE_L_STORNO && AuftragInFreigabe;
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob der Auftrag in der W4 schon als gepackt gekennzeichnet ist.
        /// </summary>
        public bool Gepackt
        {
            get
            {
                // Die neue Logik besagt, das snur noch buchbare Elemente angezeigt werden
                return false;
                foreach (var position in GetEndPositionen())
                {
                    if (position.Menge > position.BEREITS_GEBUCHT * position.BPOS_N_MENGENFAKTOR)
                    {
                        return false;
                    }
                }
                return true;
            }
        }
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob der Auftrag mindestens über eine packbare Position verfügt.
        /// </summary>
        public bool IstPackbar
        {
            get
            {
                return AuftragInFreigabe && Positionen.Any(x => x.Menge > 0);
            }
        }
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob der Auftrag vollständig in der WK5 gebucht wurde, oder nicht.
        /// </summary>
        public bool BuchenIstMöglich
        {
            get
            {
                var (gesamtMenge, bereitsGebucht) = GetMengen();
                return bereitsGebucht == gesamtMenge;
            }
        }
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob ein Teil bereits in der WK5 gebucht wurde.
        /// </summary>
        public bool TeillieferungIstMöglich
        {
            get
            {
                // HINWEIS: Stücklisten können nicht als Teillieferung vorkommen!!!!
                bool teillieferungIstMöglich = false;
                foreach (PackBelegposition position in Positionen)
                {
                    if (position.PositionIstStückliste && !position.IstGepackt && !position.HatStückListeBuchungen())
                    {
                        return false;
                    }
                    else if (position.IstBundle && position.HatBundleBuchungen())
                    {
                        bool returnValue = true;

                        // -1 Damit wir nicht nicht gebuchte Bundle als Teillieferung erlauben
                        decimal tmp = -1;
                        foreach (PackBelegposition bundlePos in position.BundleArtikel)
                        {
                            decimal bundleMenge = bundlePos.Menge / position.Menge;
                            decimal faktor = bundlePos.BEREITS_GEBUCHT / bundleMenge;

                            if(tmp == -1)
                            {
                                tmp = faktor;
                            }
                            
                            if(faktor != tmp)
                            {
                                return false;
                            }
                        }
                        return returnValue;
                    }
                    else if (!position.PositionIstStückliste && !position.IstBundle && position.BundleParent is null && position.BEREITS_GEBUCHT > 0)
                    {
                        teillieferungIstMöglich = true;
                    }
                }

                return teillieferungIstMöglich;
            }
        }
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob der Auftrag in der Packfreigabe ist, oder nicht.
        /// </summary>
        public bool AuftragInFreigabe { get; set; }
        #endregion
        #region Konstruktoren
        internal PackAuftrag()
        {

        }
        public PackAuftrag(int BELE_N_NR) : base(BELE_N_NR, "AU")
        {

        }
        #endregion
        /// <summary>
        /// Gibt an, ob der Packauftrag eine Miete enthält.
        /// </summary>
        public bool HatMiete => Positionen.Any(x => x.ARTI_L_MIETE);
        /// <summary>
        /// Generiert die Packhinweise für alle Belegpositionen
        /// </summary>
        /// <returns></returns>
        public string GetPacktext()
        {
            StringBuilder sb = new StringBuilder();

            if (!string.IsNullOrWhiteSpace(Notiz))
            {
                sb.AppendLine($"<strong>Hinweise aus Auftrag:</strong><br />");
                sb.Append(Notiz.Replace("\r\n", "<br />"));
            }

            foreach (var pos in GetEndPositionen().Where(x => x.Menge > x.BPOS_N_MENGEGELIEF && !string.IsNullOrEmpty(x.ARTI_B_PACKTEXT)))
            {
                sb.AppendLine($"<strong>({pos.Artikelnummer})</strong><br />");
                sb.AppendLine($"{pos.ARTI_B_PACKTEXT}<br /><br />");
            }

            return sb.ToString();
        }
        protected override async Task<Beleg?> LadeBelegAsync(FbController2 fbController)
        {
            fbController.AddParameter("@BELE_N_NR", Belegnummer);
            fbController.AddParameter("@BELE_A_TYP", Belegtyp);
            var belegRow = await fbController.SelectRowAsync(@"SELECT 
BELEGE.*, 
CASE WHEN EXISTS (SELECT FREI_N_BELENR FROM WK5_AUFTRAGSFREIGABE WHERE FREI_N_BELENR = BELE_N_NR AND FREI_D_PACKENAB <= CURRENT_DATE) THEN 'Y' ELSE 'N' END AS AUFTRAGINFREIGABE
FROM BELEGE 
WHERE BELE_A_TYP = @BELE_A_TYP AND BELE_N_NR = @BELE_N_NR");

            return belegRow is null ? null : ObjectErweiterung.DataRowZuObjekt(this, belegRow);
        }
        public static async Task<PackAuftrag?> GetPackAuftragAsync(int auftragsnummer, FbController2 fbController)
        {
            PackAuftrag auftrag = new PackAuftrag(auftragsnummer);
            var beleg2 = await auftrag.LadeBelegAsync(fbController);
            if (beleg2 is null)
            {
                return null;
            }
            await auftrag.LadePositionenAsync(fbController);
            return auftrag;
        }
        #region Methoden
        /// <summary>
        /// Übernimmt einen Auftrag zu einem Lieferschein
        /// </summary>
        /// <exception cref="ChargenException">Tritt auf, wenn die Positionen nicht in ausreichender Menge gechargt werden können. Wir ausgelöst durch <see cref="PrüfeVerfügbareMengen"/></exception>
        /// <returns></returns>
        public async Task<PackLieferschein> ÜbernahmeInLieferscheinAsync(FbController2 fbController, OptionCollection optionen, BelegService belegService)
        {
            // Wenn es bereits einen Lieferschein gibt, dann brauchen wir auch keinen einfügen
            //int belegnummerLieferschein = LieferscheinnummerVerknüpfung;


            if (LieferscheinnummerVerknüpfung != 0 && BELE_L_ERLEDIGT)
            {

                // AUF KEINEN FALL DEN LS ZURÜCKGEBEN!!!
                // Wenn der Auftrag auf erleidigt steht, dann soll dieser Prozess hier zwingen abstürzen!!!
                throw new InvalidOperationException("Der Auftrag ist bereits erledigt. Fortführen nicht möglich.");
                // return new PackLieferschein(LieferscheinnummerVerknüpfung);
            }

            // Hier kann eine ChargenException ausgelöst werden. Wir wollen nur den Lieferschein erstellen, wenn wir auch ausreichend Chargen können!
            await PrüfeVerfügbareMengenAsync(fbController);

            PackLieferschein lieferschein = (PackLieferschein)await belegService.ÜbernahmeAsync(this, optionen, fbController);


            // Prüfen, ob der Auftrag nun vollständig verbucht worden ist. Wenn ja, dann kann dieser erleidgt werden, ansonsten handelt es sich um eine Teillieferung.
            if (await AuftragIstVollständigGebuchtAsync(fbController))
            {
                this.BELE_A_STATUSFELD = "erledigt";
                fbController.AddParameter("@BELE_N_NR_LS", lieferschein.Belegnummer);
                fbController.AddParameter("@BELE_A_STATUSFELD", "erledigt");
                fbController.AddParameter("@BELE_A_TYP", "AU");
                fbController.AddParameter("@BELE_N_NR", Belegnummer);
                await fbController.QueryAsync("UPDATE BELEGE SET BELE_N_NR_LS = @BELE_N_NR_LS, BELE_L_ERLEDIGT = 'Y', BELE_A_STATUSFELD = @BELE_A_STATUSFELD WHERE BELE_A_TYP = @BELE_A_TYP AND BELE_N_NR = @BELE_N_NR");

                //Klein 08.04.2020 - Zur Korretkur von ausgelieferten Stücklisten Mengen
                fbController.AddParameter("@INP_AUFTRAG", Belegnummer);
                fbController.AddParameter("@INP_USER", fbController.UserId);
                await fbController.RunProcedureAsync("PROZ_AUFTRAGERLEDIGT");
            }
            else
            {
                this.BELE_A_STATUSFELD = "Teillief.";
                fbController.AddParameter("@BELE_N_NR_LS", lieferschein.Belegnummer);
                fbController.AddParameter("@BELE_A_STATUSFELD", "Teillief.");
                fbController.AddParameter("@BELE_A_TYP", "AU");
                fbController.AddParameter("@BELE_N_NR", Belegnummer);
                await fbController.QueryAsync(@"UPDATE BELEGE SET 
BELE_N_NR_LS = @BELE_N_NR_LS, 
BELE_L_ERLEDIGT = 'N', 
BELE_A_STATUSFELD = @BELE_A_STATUSFELD,
BELE_L_ABHOLUNG_INFORMIERT = 'N'
WHERE 
BELE_A_TYP = @BELE_A_TYP 
AND BELE_N_NR = @BELE_N_NR");
            }
#if !keepAuftrag
            // Auftrag aus WK5_AUFTRAGSFREIGABE löschen
            fbController.AddParameter("@BELEGNUMMER_AUFTRAG", Belegnummer);
            await fbController.QueryAsync("DELETE FROM WK5_AUFTRAGSFREIGABE WHERE FREI_N_BELENR = @BELEGNUMMER_AUFTRAG AND FREI_D_PACKENAB <= CURRENT_DATE");
#endif
            


            // Es muss gepürft werden, ob die Versandkosten doppelt berechnen werden sollen
            if (this.WK5_BELE_N_VERSAND_BERECHNET == 0 || this.VersandDoppeltBerechnen)
            {
                fbController.AddParameter("@BELEGNUMMER_AUFTRAG", Belegnummer);
                await fbController.QueryAsync("UPDATE BELEGE SET WK5_BELE_N_VERSAND_BERECHNET = WK5_BELE_N_VERSAND_BERECHNET + 1 WHERE BELE_A_TYP = 'AU' AND BELE_N_NR = @BELEGNUMMER_AUFTRAG");
            }
            else
            {
                // In allen anderen Fällen, darf der Lieferschein keine Versandkosten haben
                fbController.AddParameter("@BELEGNUMMER_LIEFERSCHEIN", lieferschein.Belegnummer);
                await fbController.QueryAsync("UPDATE BELEGE SET BELE_N_FRACHTVK = 0, BELE_N_FRACHTEK = 0 WHERE BELE_A_TYP = 'LS' AND BELE_N_NR = @BELEGNUMMER_LIEFERSCHEIN");
            }


            return lieferschein;
        }
        /// <summary>
        /// Prüft ob der Auftrag bereits vollständig ausgeliefert worden ist.
        /// <para>
        /// Ein Auftrag ist dann vollständig ausgeliefert, wenn für jede Position des Auftrages BPOS_N_MENGE kleiner als BPOS_N_MENGEGELIEF IST
        /// </para>
        /// </summary>
        /// <param name="fbController"></param>
        /// <returns></returns>
        private async Task<bool> AuftragIstVollständigGebuchtAsync(FbController2 fbController)
        {

            fbController.AddParameter("@BPOS_N_NR", Belegnummer);
            var result = await fbController.SelectRowAsync("SELECT * FROM BELEGPOS WHERE BPOS_A_TYP = 'AU' AND BPOS_N_NR = @BPOS_N_NR AND BPOS_N_MENGE != BPOS_N_MENGEGELIEF");

            return result is null;
        }
        #endregion
    }

    public sealed class PackLieferschein : PackBeleg
    {
        #region Konstruktoren
        public PackLieferschein(int BELE_N_NR) : base(BELE_N_NR, "LS")
        {

        }
        #endregion

        public static async Task<PackLieferschein?> GetPackLieferscheinAsync(int belegnummer, FbController2 fbController)
        {
            PackLieferschein lieferschein = new PackLieferschein(belegnummer);
            var beleg2 = await lieferschein.LadeBelegAsync(fbController);
            if (beleg2 is null)
            {
                return null;
            }
            await lieferschein.LadePositionenAsync(fbController);
            return lieferschein;
        }

        #region Methoden
        public async Task ChargenAsync(FbController2 fbController, AppMitarbeiter mitarbeiter)
        {
            foreach (PackBelegposition position in Positionen)
            {
                await position.ChargenAsync(fbController, mitarbeiter.PersonalNummer);
            }

            fbController.AddParameter("@BELEGNUMMER", AuftragsnummerVerknüpfung);
            fbController.AddParameter("@MITARBEITER_NAME", $"{mitarbeiter.Username}");
            await fbController.QueryAsync(@"INSERT INTO BELEG_CHANGE (BCNG_N_BELEGNR, BCNG_A_BELEGTYP, BCNG_N_ART, BCNG_A_WERTALT, BCNG_A_WERTNEU, BCNG_N_POSNR) VALUES (@BELEGNUMMER,'AU','3','Verpackt durch' , @MITARBEITER_NAME,'0')");

#if !PackenBisNichtUmstellen

            fbController.AddParameter("@BELEGNUMMER", AuftragsnummerVerknüpfung);
            await fbController.QueryAsync("UPDATE BELEGE SET BELE_TIMESTAMP = CURRENT_TIMESTAMP where bele_a_typ = 'AU' and BELE_N_NR = @BELEGNUMMER");
#endif
        }

        public async Task<Rechnung?> ÜbernahmeInRechnungAsync(FbController2 fbController, OptionCollection optionen, BelegService belegService)
        {

            if (RechnungsnummerVerknüpfung > 0)
            {
                return await Rechnung.GetRechnungAsync(RechnungsnummerVerknüpfung, fbController);
            }

            Rechnung rechnung = (Rechnung)await belegService.ÜbernahmeAsync(this, optionen, fbController);


            // Lieferschein updaten
            fbController.AddParameter("@BELE_A_STATUSFELD", "berechnet");
            fbController.AddParameter("@BELE_A_TYP", "LS");
            fbController.AddParameter("@BELE_N_NR", Belegnummer);
            await fbController.QueryAsync("update BELEGE set BELE_L_BERECHNET = 'Y', BELE_A_STATUSFELD = ? where BELE_A_TYP = ? and BELE_N_NR = ?");

            fbController.AddParameter("@INP_BELEGNR", rechnung.Belegnummer);
            fbController.AddParameter("@INP_TYP", "RE");
            fbController.AddParameter("@INP_NEW", "Y");
            fbController.AddParameter("@INP_OLD_TYP", "LS");
            fbController.AddParameter("@INP_OLD_NR", Belegnummer);
            fbController.AddParameter("@INP_UEBERNAHEMTYP", "U");
            await fbController.RunProcedureAsync("REORG_BELEGVERLAUF");

            return rechnung;
        }
        #endregion
    }
    public abstract class PackBeleg : Beleg
    {
        #region Konstruktoren
        protected PackBeleg()
        {

        }
        protected PackBeleg(int BELE_N_NR, string BELE_A_TYP) : base(BELE_A_TYP, BELE_N_NR)
        {

        }


        #endregion


        internal override async Task LadePositionenAsync(FbController2 fbController)
        {
            fbController.AddParameter("@BPOS_N_NR", Belegnummer);
            fbController.AddParameter("@BPOS_A_TYP", Belegtyp);
            var posData = await fbController.SelectDataAsync($"{Belegposition.POSITIONEN_SELECT_SQL} BPOS_N_NR = @BPOS_N_NR AND BPOS_A_TYP = @BPOS_A_TYP ORDER BY BPOS_N_POS");
            List<Belegposition> castedPositionen = new List<Belegposition>();
            foreach (DataRow row in posData.Rows)
            {
                PackBelegposition position = ObjectErweiterung.DataRowZuObjekt(new PackBelegposition(), row);
                position.BEREITS_GEBUCHT = await position.GetBereitsGebuchtAsync(fbController);
                if (position.PositionIstStückliste)
                {
                    await position.LadeStücklisteAsync(position, 0, fbController);
                }

                if (position.PositionBenötigtSeriennummer)
                {
                    await position.LadeSeriennummernAsync(position, fbController);
                }

                position.ShowWarenzugangHeuteHinweis = await position.PrüfeAufLagerzugangAsync(position, fbController);

                castedPositionen.Add(position);
            }


            int currentBundleId = 0;
            foreach (var pos in castedPositionen)
            {
                if (pos.BundleId != 0)
                {
                    if (pos.BundleId != currentBundleId)
                    {
                        currentBundleId = pos.BundleId;

                        foreach (var bundlePos in castedPositionen.Where(x => x.BundleId == currentBundleId && pos != x))
                        {
                            bundlePos.BundleParent = pos;
                            pos.BundleArtikel.Add(bundlePos);
                        }
                    }
                }

                Positionen.Add(pos);
            }

            // Wir prüfen hier, ob wir die volle Menge der Position ausliefern können, wenn nciht, dann wird die Menge auf Betsand angepasst
            foreach (PackBelegposition pos in Positionen.Where(x => x.IstBundle || (!x.IstBundle && x.BundleId == 0)))
            {
                pos.Menge -= pos.BPOS_N_MENGEGELIEF;
                if (pos.Menge > pos.Bestand && (pos.IstBundle || pos.ARTI_L_LAGERFUEHR))
                {
                    if (pos.Bestand is 0)
                    {
                        pos.Menge = 0;

                        foreach (var bundlePos in pos.BundleArtikel)
                        {
                            bundlePos.Menge = 0;
                        }
                    }
                    else
                    {
                        pos.Menge = pos.Bestand;
                    }
                }
            }
        }

        #region Positionen abrufen
        public new PackBelegposition? GetPosition(int BPOS_N_POSID)
        {
            if (Positionen == null)
            {
                return null;
            }

            var suchePosition =
                from PackBelegposition pos in Positionen
                where pos.PosId == BPOS_N_POSID
                select pos;

            return suchePosition.FirstOrDefault();
        }

        public PackBelegposition? GetPosition(PackBelegposition pos, int stücklistenPositionsId)
        {
            foreach (PackBelegposition stklPosition in pos.StücklistenArtikel)
            {
                if (stklPosition.StücklistenPositionsId == stücklistenPositionsId)
                {


                    return stklPosition;
                }

                if (stklPosition.PositionIstStückliste)
                {
                    var tmp = GetPosition(stklPosition, stücklistenPositionsId);
                    if (tmp is not null)
                    {
                        return tmp;
                    }
                }

            }

            return null;
        }
        /// <summary>
        /// Liefert alle <see cref="PackBelegposition"/>en des Beleges, die gepackt werden können.
        /// </summary>
        /// <returns></returns>
        public new IEnumerable<PackBelegposition> GetEndPositionen()
        {
            if (Positionen != null)
            {
                foreach (PackBelegposition artikel in Positionen.OrderBy(x => x.PosNr))
                {
                    if (artikel.PositionIstStückliste)
                    {
                        foreach (PackBelegposition subArtikel in GetStücklistenPositionen(artikel))
                        {
                            yield return subArtikel;
                        }
                    }
                    else
                    {
                        yield return artikel;
                    }
                }
            }
        }

        public IEnumerable<PackBelegposition> GetPositionen()
        {
            foreach (var pos in Positionen)
            {
                yield return (PackBelegposition)pos;
            }
        }
        /// <summary>
        /// Liefert alle Artikel einer Stückliste
        /// </summary>
        /// <param name="pos"></param>
        /// <returns></returns>
        private IEnumerable<PackBelegposition> GetStücklistenPositionen(PackBelegposition pos)
        {
            foreach (PackBelegposition artikel in pos.StücklistenArtikel)
            {
                // Es kann ggf. vorkommen, dass eine Stückliste eine Stückliste enthält. Wir müssen also rekursiv an das ganze herangehen.
                if (artikel.PositionIstStückliste)
                {
                    foreach (PackBelegposition subArtikel in GetStücklistenPositionen(artikel))
                    {
                        yield return subArtikel;
                    }
                }
                else
                {
                    yield return artikel;
                }
            }
        }
        /// <summary>
        /// Liefert alle verschiedenen Unterwarengruppen der Belegpositionen.
        /// </summary>
        /// <returns></returns>
        public List<string> GetUnterwarengruppen()
        {
            List<string> unterwarengruppen = new List<string>();

            foreach (var artikel in GetEndPositionen())
            {
                if (!String.IsNullOrWhiteSpace(artikel.ARTI_A_UNTERWARENG) && !unterwarengruppen.Contains(artikel.ARTI_A_UNTERWARENG))
                {
                    unterwarengruppen.Add(artikel.ARTI_A_UNTERWARENG);
                }
            }

            return unterwarengruppen;
        }
        #endregion
        #region Mengen Überprüfung
        /// <summary>
        /// Die Methode liefert die Anzahl aller Artikel, die von der WK5 gebucht werden müssen, sowie die bereits gebuchte Anzahl an Artikel aus der WK5.
        /// </summary>
        /// <returns></returns>
        protected (decimal gesamtMenge, decimal bereitsGebucht) GetMengen()
        {
            // HINWEIS: Stücklisten können nicht als Teillieferung versendet werden.
            (decimal gesamtMenge, decimal bereitsGebucht) result = (0, 0);

            foreach (var artikel in this.GetEndPositionen())
            {
                // Die Menge ist bereits abzüglich der bereits gelieferten Menge
                result = (result.gesamtMenge += artikel.Menge, result.bereitsGebucht += artikel.BEREITS_GEBUCHT);
            }

            return result;
        }
        /// <summary>
        /// Die Methode überprüft, ob für alle <see cref="PackBelegposition"/>en für den Wert <see cref="PackBelegposition.BEREITS_GEBUCHT"/> noch genügend Chargen in der W4 vorhanden sind.
        /// </summary>
        /// <exception cref="ChargenException">Wird ausgelöst, wenn für einen Artikel nicht ausreichend Chargen verfügbar sind.</exception>
        protected async Task PrüfeVerfügbareMengenAsync(FbController2 fbController)
        {
            foreach (var position in GetEndPositionen().Where(pos => pos.ARTI_L_LAGERFUEHR))
            {

                fbController.AddParameter("@ARTIKELNUMMER", position.Artikelnummer);
                var data = await fbController.SelectRowAsync("SELECT SUM(CHAR_N_MENGEOFFEN) AS MENGE FROM CHARGEN WHERE CHAR_A_ARTINR = @ARTIKELNUMMER AND CHAR_N_MENGEOFFEN > 0");

                if (data is null)
                {
                    throw new ChargenException(position.Artikelnummer, 0);
                }

                _ = decimal.TryParse(data["MENGE"].ToString(), out decimal MENGE_CHARGEN_VERFÜGBAR);

                // Es müssen genügend Chargen vorhanden sein, um alle in der WK5 gebuchten Positionen im Lieferschein zu Chargen.
                if (MENGE_CHARGEN_VERFÜGBAR < 0 || MENGE_CHARGEN_VERFÜGBAR < position.BEREITS_GEBUCHT)
                {
                    throw new ChargenException(position.Artikelnummer, MENGE_CHARGEN_VERFÜGBAR);
                }
            }
        }
        #endregion

        public override decimal GetBelegGewicht(bool alternativUndEventualAbziehen = false)
        {
            decimal gewicht = 0;
            foreach (var item in GetEndPositionen())
            {
                if ((item.Option.Equals("Alternativposition") || item.Option.Equals("Eventualposition")) && alternativUndEventualAbziehen)
                    continue;
                gewicht += item.Gewicht * item.Menge;
            }
            return gewicht;
        }
    }
    /// <summary>
    /// Stellt einen bereits gepackten Auftrag dar.
    /// </summary>
    public sealed class AuftragGepackt
    {
        public int Id { get; set; }
        [CompareField("GEPACKT_N_BELENR")]
        public int Auftragsnummer { get; set; }
        [CompareField("GEPACKT_N_LS_NR")]
        public int Lieferscheinnummer { get; set; }
        [CompareField("GEPACKT_N_RE_NR")]
        public int Rechnungsnummer { get; set; }
        [CompareField("BELE_D_DATE")]
        public DateTime Belegdatum { get; set; }
        [CompareField("BELE_A_NAME1")]
        public string Name1 { get; set; } = string.Empty;
        [CompareField("BELE_A_NAME2")]
        public string Name2 { get; set; } = string.Empty;
    }
}
