﻿using KarleyLibrary.Attributes;
using KarleyLibrary.Erweiterungen;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace WK5.Core.Models
{
    public class Lieferantenartikel
    {
        private string _lieferantenName = string.Empty;
        private string _artikelnummer = String.Empty;
        private decimal _preis;
        private int _mindestabnahme = 1;
        #region DB Props
        public string ARLI_A_ARTIKELNR { get => _artikelnummer; set => _artikelnummer = (value ?? string.Empty).ToUpper(); }
        public int ARLI_N_LIEFNR { get; set; }
        public string? ARLI_A_BESTELLNR { get; set; }
        public bool ARLI_L_HAUPTLIEFERANT { get; set; }
        public decimal ARLI_N_PREIS
        {
            get => _preis;
            set
            {
                _preis = value;
                Staffelpreise.SetBezugspreis(_preis);
            }
        }
        public decimal ARLI_N_RABATT { get; set; }
        public decimal ARLI_N_RABATT2 { get; set; }
        public int ARLI_N_MINDESTABNAHME { get => _mindestabnahme; set => _mindestabnahme = value < 1 ? 1 : value; }
        public string? ARLI_A_WAEHRUNG { get; set; } = "EUR";
        public decimal ARLI_N_WAEHRUNGSPREIS { get; set; }
        public decimal ARLI_N_KURS { get; set; }
        public string? ARLI_A_EINHEIT { get; set; }
        public decimal ARLI_N_UMRECHNUNG { get; set; }
        public decimal ARLI_N_PREISFREMDEINHEIT { get; set; }
        public int ARLI_STAFFELART { get; set; }
        public decimal ARLI_N_MENGE1 { get; set; }
        public decimal ARLI_N_PREISMENGE1 { get; set; }
        public decimal ARLI_N_RABATTMENGE1 { get; set; }
        public decimal ARLI_N_MENGE2 { get; set; }
        public decimal ARLI_N_PREISMENGE2 { get; set; }
        public decimal ARLI_N_RABATTMENGE2 { get; set; }
        public decimal ARLI_N_MENGE3 { get; set; }
        public decimal ARLI_N_PREISMENGE3 { get; set; }
        public decimal ARLI_N_RABATTMENGE3 { get; set; }
        public decimal ARLI_N_MENGE4 { get; set; }
        public decimal ARLI_N_PREISMENGE4 { get; set; }
        public decimal ARLI_N_RABATTMENGE4 { get; set; }
        public decimal ARLI_N_VERSANDKOST { get; set; }
        public decimal ARLI_N_VERPACKKOST { get; set; }
        public decimal ARLI_N_SONSTKOST { get; set; }
        public decimal ARLI_N_TEUERUNG { get; set; }
        public DateTime ARLI_D_LETZAEND { get; set; }
        public string? ARLI_B_NOTIZ { get; set; }
        public string? ARLI_A_BEZ1 { get; set; }
        public string? ARLI_A_BEZ2 { get; set; }
        public string? ARLI_A_BEZ3 { get; set; }
        public string? ARLI_A_BEZ4 { get; set; }
        public string? ARLI_A_BEZ5 { get; set; }
        public int ARLI_N_LASTUSER { get; set; }
        public DateTime ARLI_TIMESTAMP { get; set; }
        public decimal WK5_ARLI_N_LIEFBESTAND { get; set; }
        public bool ARLI_L_NURVIELFACHES { get; set; }

        [CompareField("LIEF_A_NAME1")]
        public string LieferantenName { get => _lieferantenName; set => _lieferantenName = value ?? string.Empty; }

        public StaffelpreisCollection Staffelpreise { get; private set; } = new StaffelpreisCollection(4);
        #endregion
        public Lieferantenartikel()
        {
            InitializeStaffelpreisCollection();
        }
        private void InitializeStaffelpreisCollection()
        {
            StaffelpreisCollection staffelCollection = new StaffelpreisCollection(4);
            staffelCollection.SetBezugspreis(ARLI_N_PREIS);
            staffelCollection.Add(ARLI_N_MENGE1, ARLI_N_PREISMENGE1, ARLI_N_RABATTMENGE1);
            staffelCollection.Add(ARLI_N_MENGE2, ARLI_N_PREISMENGE2, ARLI_N_RABATTMENGE2);
            staffelCollection.Add(ARLI_N_MENGE3, ARLI_N_PREISMENGE3, ARLI_N_RABATTMENGE3);
            staffelCollection.Add(ARLI_N_MENGE4, ARLI_N_PREISMENGE4, ARLI_N_RABATTMENGE4);

            Staffelpreise = staffelCollection;
        }

        private static void ConvertStaffelpreisCollection(Lieferantenartikel artikel)
        {
            artikel.Staffelpreise.Clean();
            artikel.Staffelpreise.Sort();
            for (int i = 0; i < artikel.Staffelpreise.MaxAnzahl; i++)
            {
                Staffelpreis? preis = artikel.Staffelpreise.Get(i);
                switch (i)
                {
                    case 0:
                        {
                            artikel.ARLI_N_PREISMENGE1 = artikel.ARLI_N_PREIS;
                            artikel.ARLI_N_MENGE1 = preis?.Menge ?? 0;
                            artikel.ARLI_N_RABATTMENGE1 = preis?.Rabatt ?? 0;
                        }
                        break;
                    case 1:
                        {
                            artikel.ARLI_N_PREISMENGE2 = artikel.ARLI_N_PREIS;
                            artikel.ARLI_N_MENGE2 = preis?.Menge ?? 0;
                            artikel.ARLI_N_RABATTMENGE2 = preis?.Rabatt ?? 0;
                        }
                        break;
                    case 2:
                        {
                            artikel.ARLI_N_PREISMENGE3 = artikel.ARLI_N_PREIS;
                            artikel.ARLI_N_MENGE3 = preis?.Menge ?? 0;
                            artikel.ARLI_N_RABATTMENGE3 = preis?.Rabatt ?? 0;
                        }
                        break;
                    case 3:
                        {
                            artikel.ARLI_N_PREISMENGE4 = artikel.ARLI_N_PREIS;
                            artikel.ARLI_N_MENGE4 = preis?.Menge ?? 0;
                            artikel.ARLI_N_RABATTMENGE4 = preis?.Rabatt ?? 0;
                        }
                        break;
                    default:
                        break;
                }
            }

        }
        public static async IAsyncEnumerable<Lieferantenartikel> GetLieferantenartikelAsync(string ARLI_A_ARTIKELNR, FbController2 fbController)
        {
            if (ARLI_A_ARTIKELNR is not null)
            {
                fbController.AddParameter("@ARLI_A_ARTIKELNR", ARLI_A_ARTIKELNR.ToUpper());
                var data = await fbController.SelectDataAsync(@"SELECT 
LFA.*,
LIEF_A_NAME1
FROM LIEFERANTENARTIKEL LFA 
LEFT JOIN LIEFERANTEN L ON L.LIEF_N_NR = LFA.ARLI_N_LIEFNR
WHERE UPPER(ARLI_A_ARTIKELNR) = @ARLI_A_ARTIKELNR");

                foreach (DataRow row in data.Rows)
                {
                    var artikel = ObjectErweiterung.DataRowZuObjekt(new Lieferantenartikel(), row);
                    artikel.InitializeStaffelpreisCollection();
                    yield return artikel;
                }
            }
        }


        public static async Task<Lieferantenartikel?> GetLieferantenartikelAsync(string artikelnummer, int lieferantenNummer, FbController2 fbController)
        {
            fbController.AddParameter("@ARLI_A_ARTIKELNR", artikelnummer.ToUpper());
            fbController.AddParameter("@ARLI_N_LIEFNR", lieferantenNummer);

            DataRow? row = await fbController.SelectRowAsync(@"SELECT 
LFA.*,
LIEF_A_NAME1
FROM LIEFERANTENARTIKEL LFA 
LEFT JOIN LIEFERANTEN L ON L.LIEF_N_NR = LFA.ARLI_N_LIEFNR
WHERE UPPER(ARLI_A_ARTIKELNR) = @ARLI_A_ARTIKELNR AND ARLI_N_LIEFNR = @ARLI_N_LIEFNR");

            if (row is null)
            {
                return null;
            }
            else
            {
                var artikel = ObjectErweiterung.DataRowZuObjekt(new Lieferantenartikel(), row);
                artikel.InitializeStaffelpreisCollection();
                return artikel;
            }
        }

        public static async IAsyncEnumerable<Lieferantenartikel> GetLieferantenartikelAsync(int lieferantenNummer, FbController2 fbController)
        {
            fbController.AddParameter("@ARLI_N_LIEFNR", lieferantenNummer);
            DataTable data = await fbController.SelectDataAsync(@"SELECT 
LFA.*,
LIEF_A_NAME1
FROM LIEFERANTENARTIKEL LFA 
LEFT JOIN LIEFERANTEN L ON L.LIEF_N_NR = LFA.ARLI_N_LIEFNR
WHERE ARLI_N_LIEFNR = @ARLI_N_LIEFNR");

            foreach(DataRow row in data.Rows)
            {
                Lieferantenartikel arli = ObjectErweiterung.DataRowZuObjekt(new Lieferantenartikel(), row);
                arli.InitializeStaffelpreisCollection();
                yield return arli;
            }
        }

        public string GetBezeichnung() => Regex.Replace($"{ARLI_A_BEZ1} {ARLI_A_BEZ2} {ARLI_A_BEZ3} {ARLI_A_BEZ4} {ARLI_A_BEZ5}", "  +/g", " ").Trim();


        /// <summary>
        /// Liefert den Preis abzüglich aller Rabatte.
        /// <para>
        /// Staffelpreise werden ebenfalls berücksichtigt.
        /// </para>
        /// </summary>
        public decimal PreisMitRabatt(decimal menge)
        {
            if (ARLI_N_MENGE4 != 0 && ARLI_N_MENGE4 <= menge)
            {
                return ARLI_N_PREIS * (100 - ARLI_N_RABATTMENGE4) / 100;
            }
            else if (ARLI_N_MENGE3 != 0 && ARLI_N_MENGE3 <= menge)
            {
                return ARLI_N_PREIS * (100 - ARLI_N_RABATTMENGE3) / 100;

            }
            else if (ARLI_N_MENGE2 != 0 && ARLI_N_MENGE2 <= menge)
            {
                return ARLI_N_PREIS * (100 - ARLI_N_RABATTMENGE2) / 100;
            }
            else if (ARLI_N_MENGE1 != 0 && ARLI_N_MENGE1 <= menge)
            {
                return ARLI_N_PREIS * (100 - ARLI_N_RABATTMENGE1) / 100;
            }
            else
            {
                // Kein Staffelpreis gefunden, also rechnen wir auf Basis des normalen Preises und Rabattes.
                decimal preis = ARLI_N_PREIS;

                if (ARLI_N_RABATT > 0)
                {
                    preis *= (100 - ARLI_N_RABATT) / 100;
                }

                if (ARLI_N_RABATT2 > 0)
                {
                    preis *= (100 - ARLI_N_RABATT2) / 100;
                }

                return preis;
            }
        }

        public decimal GetPreis(decimal menge)
        {
            var (staffelMenge, staffelPreis, _) = GetStaffelpreis(menge);

            // Default von decimal ist 0, wenn also 0, dann gibt es keine Staffelmenge.
            return staffelMenge > 0 ? staffelPreis : ARLI_N_PREIS;
        }

        /// <summary>
        /// Prüft ob es für die übergebene Menge einen Staffelpreis gibt und gibt diesen zurück.
        /// </summary>
        /// <param name="menge"></param>
        /// <returns>Wenn kein Staffelpreis gefunden wurde, dann ist staffelMenge 0. Ansonsten liefert die Funktion ein Tuple mit der Menge, dem Preis und Rabatt zurück.</returns>
        private (decimal staffelMenge, decimal staffelPreis, decimal staffelRabatt) GetStaffelpreis(decimal menge)
        {
            // Die Staffelpreise können in der W4 in einer falschen Reihenfolge hinterlegt sein, daher packen wir alle Staffelpreise in ein Array und sortieren manuell
            (decimal staffelMenge, decimal staffelPreis, decimal staffelRabatt)[] staffelpreise = new (decimal, decimal, decimal)[]
            {
                (ARLI_N_MENGE1, ARLI_N_PREIS, ARLI_N_RABATTMENGE1),
                (ARLI_N_MENGE2, ARLI_N_PREIS, ARLI_N_RABATTMENGE2),
                (ARLI_N_MENGE3, ARLI_N_PREIS, ARLI_N_RABATTMENGE3),
                (ARLI_N_MENGE4, ARLI_N_PREIS, ARLI_N_RABATTMENGE4)
            };

            var sucheStaffelpreis = staffelpreise.Where(x => x.staffelMenge > 0 && x.staffelMenge <= menge).OrderByDescending(x => x.staffelMenge);
            return sucheStaffelpreis.FirstOrDefault();
        }

        /// <summary>
        /// Holt die Rabatte für den Lieferantenartikel in Abhängigkeit der Bestellmenge zur Staffelmenge
        /// </summary>
        /// <param name="menge"></param>
        /// <returns></returns>
        public (decimal rabatt1, decimal rabatt2) GetRabatte(decimal menge)
        {
            var (staffelMenge, _, staffelRabatt) = GetStaffelpreis(menge);
            // Wenn es einen Staffelpreis gibt, dann ist Rabatt2 immer 0, da wir auf den Staffelpreis keinen zweiten Rabatt gewähren.
            return staffelMenge > 0 ? (staffelRabatt, 0) : (ARLI_N_RABATT, ARLI_N_RABATT2);
        }

        public override bool Equals(object? obj)
        {
            return obj is Lieferantenartikel lieferantenartikel &&
                ARLI_A_ARTIKELNR == lieferantenartikel.ARLI_A_ARTIKELNR &&
                ARLI_N_LIEFNR == lieferantenartikel.ARLI_N_LIEFNR &&
                ARLI_A_BESTELLNR == lieferantenartikel.ARLI_A_BESTELLNR &&
                ARLI_L_HAUPTLIEFERANT == lieferantenartikel.ARLI_L_HAUPTLIEFERANT &&
                ARLI_N_PREIS == lieferantenartikel.ARLI_N_PREIS &&
                ARLI_N_RABATT == lieferantenartikel.ARLI_N_RABATT &&
                ARLI_N_RABATT2 == lieferantenartikel.ARLI_N_RABATT2 &&
                ARLI_N_MINDESTABNAHME == lieferantenartikel.ARLI_N_MINDESTABNAHME &&
                ARLI_A_WAEHRUNG == lieferantenartikel.ARLI_A_WAEHRUNG &&
                ARLI_N_WAEHRUNGSPREIS == lieferantenartikel.ARLI_N_WAEHRUNGSPREIS &&
                ARLI_N_KURS == lieferantenartikel.ARLI_N_KURS &&
                ARLI_A_EINHEIT == lieferantenartikel.ARLI_A_EINHEIT &&
                ARLI_N_UMRECHNUNG == lieferantenartikel.ARLI_N_UMRECHNUNG &&
                ARLI_N_PREISFREMDEINHEIT == lieferantenartikel.ARLI_N_PREISFREMDEINHEIT &&
                ARLI_STAFFELART == lieferantenartikel.ARLI_STAFFELART &&
                ARLI_N_MENGE1 == lieferantenartikel.ARLI_N_MENGE1 &&
                ARLI_N_PREISMENGE1 == lieferantenartikel.ARLI_N_PREISMENGE1 &&
                ARLI_N_RABATTMENGE1 == lieferantenartikel.ARLI_N_RABATTMENGE1 &&
                ARLI_N_MENGE2 == lieferantenartikel.ARLI_N_MENGE2 &&
                ARLI_N_PREISMENGE2 == lieferantenartikel.ARLI_N_PREISMENGE2 &&
                ARLI_N_RABATTMENGE2 == lieferantenartikel.ARLI_N_RABATTMENGE2 &&
                ARLI_N_MENGE3 == lieferantenartikel.ARLI_N_MENGE3 &&
                ARLI_N_PREISMENGE3 == lieferantenartikel.ARLI_N_PREISMENGE3 &&
                ARLI_N_RABATTMENGE3 == lieferantenartikel.ARLI_N_RABATTMENGE3 &&
                ARLI_N_MENGE4 == lieferantenartikel.ARLI_N_MENGE4 &&
                ARLI_N_PREISMENGE4 == lieferantenartikel.ARLI_N_PREISMENGE4 &&
                ARLI_N_RABATTMENGE4 == lieferantenartikel.ARLI_N_RABATTMENGE4 &&
                ARLI_N_VERSANDKOST == lieferantenartikel.ARLI_N_VERSANDKOST &&
                ARLI_N_VERPACKKOST == lieferantenartikel.ARLI_N_VERPACKKOST &&
                ARLI_N_SONSTKOST == lieferantenartikel.ARLI_N_SONSTKOST &&
                ARLI_N_TEUERUNG == lieferantenartikel.ARLI_N_TEUERUNG &&
                ARLI_D_LETZAEND == lieferantenartikel.ARLI_D_LETZAEND &&
                ARLI_B_NOTIZ == lieferantenartikel.ARLI_B_NOTIZ &&
                ARLI_A_BEZ1 == lieferantenartikel.ARLI_A_BEZ1 &&
                ARLI_A_BEZ2 == lieferantenartikel.ARLI_A_BEZ2 &&
                ARLI_A_BEZ3 == lieferantenartikel.ARLI_A_BEZ3 &&
                ARLI_A_BEZ4 == lieferantenartikel.ARLI_A_BEZ4 &&
                ARLI_A_BEZ5 == lieferantenartikel.ARLI_A_BEZ5 &&
                ARLI_N_LASTUSER == lieferantenartikel.ARLI_N_LASTUSER &&
                ARLI_TIMESTAMP == lieferantenartikel.ARLI_TIMESTAMP &&
                WK5_ARLI_N_LIEFBESTAND == lieferantenartikel.WK5_ARLI_N_LIEFBESTAND;
        }

        public override int GetHashCode()
        {
            return HashCode.Combine(ARLI_A_ARTIKELNR, ARLI_N_LIEFNR);
        }

        public static Task CreateLieferantenArtikelAsync(Lieferantenartikel artikel, FbController2 fbController)
        {
            ConvertStaffelpreisCollection(artikel);
            string sql = @"INSERT INTO LIEFERANTENARTIKEL
(
ARLI_A_ARTIKELNR, 
ARLI_N_LIEFNR, 
ARLI_A_BESTELLNR, 
ARLI_L_HAUPTLIEFERANT, 
ARLI_N_PREIS, 
ARLI_N_RABATT, 
ARLI_N_RABATT2, 
ARLI_N_MINDESTABNAHME, 
ARLI_A_WAEHRUNG, 
ARLI_N_WAEHRUNGSPREIS, 
ARLI_N_KURS, 
ARLI_A_EINHEIT, 
ARLI_N_UMRECHNUNG, 
ARLI_N_PREISFREMDEINHEIT, 
ARLI_STAFFELART, 
ARLI_N_MENGE1, 
ARLI_N_RABATTMENGE1, 
ARLI_N_MENGE2, 
ARLI_N_RABATTMENGE2, 
ARLI_N_MENGE3, 
ARLI_N_RABATTMENGE3, 
ARLI_N_MENGE4, 
ARLI_N_RABATTMENGE4, 
ARLI_N_VERSANDKOST, 
ARLI_N_VERPACKKOST, 
ARLI_N_SONSTKOST,
ARLI_N_TEUERUNG, 
ARLI_D_LETZAEND, 
ARLI_B_NOTIZ, 
ARLI_A_BEZ1, 
ARLI_A_BEZ2, 
ARLI_A_BEZ3, 
ARLI_A_BEZ4, 
ARLI_A_BEZ5, 
ARLI_N_LASTUSER, 
ARLI_TIMESTAMP, 
WK5_ARLI_N_LIEFBESTAND,
ARLI_L_NURVIELFACHES
)
VALUES
(
@ARLI_A_ARTIKELNR,
@ARLI_N_LIEFNR,
@ARLI_A_BESTELLNR,
@ARLI_L_HAUPTLIEFERANT,
@ARLI_N_PREIS,
@ARLI_N_RABATT,
@ARLI_N_RABATT2,
@ARLI_N_MINDESTABNAHME,
@ARLI_A_WAEHRUNG,
@ARLI_N_WAEHRUNGSPREIS,
@ARLI_N_KURS,
@ARLI_A_EINHEIT,
@ARLI_N_UMRECHNUNG,
@ARLI_N_PREISFREMDEINHEIT,
@ARLI_STAFFELART,
@ARLI_N_MENGE1,
@ARLI_N_RABATTMENGE1,
@ARLI_N_MENGE2,
@ARLI_N_RABATTMENGE2,
@ARLI_N_MENGE3,
@ARLI_N_RABATTMENGE3,
@ARLI_N_MENGE4,
@ARLI_N_RABATTMENGE4,
@ARLI_N_VERSANDKOST,
@ARLI_N_VERPACKKOST,
@ARLI_N_SONSTKOST,
@ARLI_N_TEUERUNG,
@ARLI_D_LETZAEND,
@ARLI_B_NOTIZ,
@ARLI_A_BEZ1,
@ARLI_A_BEZ2,
@ARLI_A_BEZ3,
@ARLI_A_BEZ4,
@ARLI_A_BEZ5,
@ARLI_N_LASTUSER,
@ARLI_TIMESTAMP,
@WK5_ARLI_N_LIEFBESTAND,
@ARLI_L_NURVIELFACHES
)";
            fbController.AddParameter("@ARLI_A_ARTIKELNR", artikel.ARLI_A_ARTIKELNR);
            fbController.AddParameter("@ARLI_N_LIEFNR", artikel.ARLI_N_LIEFNR);
            fbController.AddParameter("@ARLI_A_BESTELLNR", artikel.ARLI_A_BESTELLNR);
            fbController.AddParameter("@ARLI_L_HAUPTLIEFERANT", artikel.ARLI_L_HAUPTLIEFERANT);
            fbController.AddParameter("@ARLI_N_PREIS", artikel.ARLI_N_PREIS);
            fbController.AddParameter("@ARLI_N_RABATT", artikel.ARLI_N_RABATT);
            fbController.AddParameter("@ARLI_N_RABATT2", artikel.ARLI_N_RABATT2);
            fbController.AddParameter("@ARLI_N_MINDESTABNAHME", artikel.ARLI_N_MINDESTABNAHME);
            fbController.AddParameter("@ARLI_A_WAEHRUNG", artikel.ARLI_A_WAEHRUNG);
            fbController.AddParameter("@ARLI_N_WAEHRUNGSPREIS", artikel.ARLI_N_WAEHRUNGSPREIS);
            fbController.AddParameter("@ARLI_N_KURS", artikel.ARLI_N_KURS);
            fbController.AddParameter("@ARLI_A_EINHEIT", artikel.ARLI_A_EINHEIT);
            fbController.AddParameter("@ARLI_N_UMRECHNUNG", artikel.ARLI_N_UMRECHNUNG);
            fbController.AddParameter("@ARLI_N_PREISFREMDEINHEIT", artikel.ARLI_N_PREISFREMDEINHEIT);
            fbController.AddParameter("@ARLI_STAFFELART", artikel.ARLI_STAFFELART);
            fbController.AddParameter("@ARLI_N_MENGE1", artikel.ARLI_N_MENGE1);
            fbController.AddParameter("@ARLI_N_RABATTMENGE1", artikel.ARLI_N_RABATTMENGE1);
            fbController.AddParameter("@ARLI_N_MENGE2", artikel.ARLI_N_MENGE2);
            fbController.AddParameter("@ARLI_N_RABATTMENGE2", artikel.ARLI_N_RABATTMENGE2);
            fbController.AddParameter("@ARLI_N_MENGE3", artikel.ARLI_N_MENGE3);
            fbController.AddParameter("@ARLI_N_RABATTMENGE3", artikel.ARLI_N_RABATTMENGE3);
            fbController.AddParameter("@ARLI_N_MENGE4", artikel.ARLI_N_MENGE4);
            fbController.AddParameter("@ARLI_N_RABATTMENGE4", artikel.ARLI_N_RABATTMENGE4);
            fbController.AddParameter("@ARLI_N_VERSANDKOST", artikel.ARLI_N_VERSANDKOST);
            fbController.AddParameter("@ARLI_N_VERPACKKOST", artikel.ARLI_N_VERPACKKOST);
            fbController.AddParameter("@ARLI_N_SONSTKOST", artikel.ARLI_N_SONSTKOST);
            fbController.AddParameter("@ARLI_N_TEUERUNG", artikel.ARLI_N_TEUERUNG);
            fbController.AddParameter("@ARLI_D_LETZAEND", artikel.ARLI_D_LETZAEND);
            fbController.AddParameter("@ARLI_B_NOTIZ", artikel.ARLI_B_NOTIZ);
            fbController.AddParameter("@ARLI_A_BEZ1", artikel.ARLI_A_BEZ1);
            fbController.AddParameter("@ARLI_A_BEZ2", artikel.ARLI_A_BEZ2);
            fbController.AddParameter("@ARLI_A_BEZ3", artikel.ARLI_A_BEZ3);
            fbController.AddParameter("@ARLI_A_BEZ4", artikel.ARLI_A_BEZ4);
            fbController.AddParameter("@ARLI_A_BEZ5", artikel.ARLI_A_BEZ5);
            fbController.AddParameter("@ARLI_N_LASTUSER", artikel.ARLI_N_LASTUSER);
            fbController.AddParameter("@ARLI_TIMESTAMP", artikel.ARLI_TIMESTAMP);
            fbController.AddParameter("@WK5_ARLI_N_LIEFBESTAND", artikel.WK5_ARLI_N_LIEFBESTAND);
            fbController.AddParameter("@ARLI_L_NURVIELFACHES", artikel.ARLI_L_NURVIELFACHES);
            return fbController.QueryAsync(sql);
        }

        public static Task UpdateLieferantenArtikelAsync(Lieferantenartikel artikel, FbController2 fbController)
        {
            ConvertStaffelpreisCollection(artikel);
            string sql = @"UPDATE LIEFERANTENARTIKEL SET
ARLI_A_BESTELLNR = @ARLI_A_BESTELLNR,
ARLI_L_HAUPTLIEFERANT = @ARLI_L_HAUPTLIEFERANT,
ARLI_N_PREIS = @ARLI_N_PREIS,
ARLI_N_RABATT = @ARLI_N_RABATT,
ARLI_N_RABATT2 = @ARLI_N_RABATT2,
ARLI_N_MINDESTABNAHME = @ARLI_N_MINDESTABNAHME,
ARLI_A_WAEHRUNG = @ARLI_A_WAEHRUNG,
ARLI_N_WAEHRUNGSPREIS = @ARLI_N_WAEHRUNGSPREIS,
ARLI_N_KURS = @ARLI_N_KURS,
ARLI_A_EINHEIT = @ARLI_A_EINHEIT,
ARLI_N_UMRECHNUNG = @ARLI_N_UMRECHNUNG,
ARLI_N_PREISFREMDEINHEIT = @ARLI_N_PREISFREMDEINHEIT,
ARLI_STAFFELART = @ARLI_STAFFELART,
ARLI_N_MENGE1 = @ARLI_N_MENGE1,
ARLI_N_RABATTMENGE1 = @ARLI_N_RABATTMENGE1,
ARLI_N_MENGE2 = @ARLI_N_MENGE2,
ARLI_N_RABATTMENGE2 = @ARLI_N_RABATTMENGE2,
ARLI_N_MENGE3 = @ARLI_N_MENGE3,
ARLI_N_RABATTMENGE3 = @ARLI_N_RABATTMENGE3,
ARLI_N_MENGE4 = @ARLI_N_MENGE4,
ARLI_N_RABATTMENGE4 = @ARLI_N_RABATTMENGE4,
ARLI_N_VERSANDKOST = @ARLI_N_VERSANDKOST,
ARLI_N_VERPACKKOST = @ARLI_N_VERPACKKOST,
ARLI_N_SONSTKOST = @ARLI_N_SONSTKOST,
ARLI_N_TEUERUNG = @ARLI_N_TEUERUNG,
ARLI_D_LETZAEND = @ARLI_D_LETZAEND,
ARLI_B_NOTIZ = @ARLI_B_NOTIZ,
ARLI_A_BEZ1 = @ARLI_A_BEZ1,
ARLI_A_BEZ2 = @ARLI_A_BEZ2,
ARLI_A_BEZ3 = @ARLI_A_BEZ3,
ARLI_A_BEZ4 = @ARLI_A_BEZ4,
ARLI_A_BEZ5 = @ARLI_A_BEZ5,
ARLI_N_LASTUSER = @ARLI_N_LASTUSER,
ARLI_TIMESTAMP = @ARLI_TIMESTAMP,
WK5_ARLI_N_LIEFBESTAND = @WK5_ARLI_N_LIEFBESTAND,
ARLI_L_NURVIELFACHES = @ARLI_L_NURVIELFACHES

WHERE UPPER(ARLI_A_ARTIKELNR) = @ARLI_A_ARTIKELNR AND ARLI_N_LIEFNR = @ARLI_N_LIEFNR";

            fbController.AddParameter("@ARLI_A_ARTIKELNR", artikel.ARLI_A_ARTIKELNR);
            fbController.AddParameter("@ARLI_N_LIEFNR", artikel.ARLI_N_LIEFNR);
            fbController.AddParameter("@ARLI_A_BESTELLNR", artikel.ARLI_A_BESTELLNR);
            fbController.AddParameter("@ARLI_L_HAUPTLIEFERANT", artikel.ARLI_L_HAUPTLIEFERANT);
            fbController.AddParameter("@ARLI_N_PREIS", artikel.ARLI_N_PREIS);
            fbController.AddParameter("@ARLI_N_RABATT", artikel.ARLI_N_RABATT);
            fbController.AddParameter("@ARLI_N_RABATT2", artikel.ARLI_N_RABATT2);
            fbController.AddParameter("@ARLI_N_MINDESTABNAHME", artikel.ARLI_N_MINDESTABNAHME);
            fbController.AddParameter("@ARLI_A_WAEHRUNG", artikel.ARLI_A_WAEHRUNG);
            fbController.AddParameter("@ARLI_N_WAEHRUNGSPREIS", artikel.ARLI_N_WAEHRUNGSPREIS);
            fbController.AddParameter("@ARLI_N_KURS", artikel.ARLI_N_KURS);
            fbController.AddParameter("@ARLI_A_EINHEIT", artikel.ARLI_A_EINHEIT);
            fbController.AddParameter("@ARLI_N_UMRECHNUNG", artikel.ARLI_N_UMRECHNUNG);
            fbController.AddParameter("@ARLI_N_PREISFREMDEINHEIT", artikel.ARLI_N_PREISFREMDEINHEIT);
            fbController.AddParameter("@ARLI_STAFFELART", artikel.ARLI_STAFFELART);
            fbController.AddParameter("@ARLI_N_MENGE1", artikel.ARLI_N_MENGE1);
            fbController.AddParameter("@ARLI_N_RABATTMENGE1", artikel.ARLI_N_RABATTMENGE1);
            fbController.AddParameter("@ARLI_N_MENGE2", artikel.ARLI_N_MENGE2);
            fbController.AddParameter("@ARLI_N_RABATTMENGE2", artikel.ARLI_N_RABATTMENGE2);
            fbController.AddParameter("@ARLI_N_MENGE3", artikel.ARLI_N_MENGE3);
            fbController.AddParameter("@ARLI_N_RABATTMENGE3", artikel.ARLI_N_RABATTMENGE3);
            fbController.AddParameter("@ARLI_N_MENGE4", artikel.ARLI_N_MENGE4);
            fbController.AddParameter("@ARLI_N_RABATTMENGE4", artikel.ARLI_N_RABATTMENGE4);
            fbController.AddParameter("@ARLI_N_VERSANDKOST", artikel.ARLI_N_VERSANDKOST);
            fbController.AddParameter("@ARLI_N_VERPACKKOST", artikel.ARLI_N_VERPACKKOST);
            fbController.AddParameter("@ARLI_N_SONSTKOST", artikel.ARLI_N_SONSTKOST);
            fbController.AddParameter("@ARLI_N_TEUERUNG", artikel.ARLI_N_TEUERUNG);
            fbController.AddParameter("@ARLI_D_LETZAEND", artikel.ARLI_D_LETZAEND);
            fbController.AddParameter("@ARLI_B_NOTIZ", artikel.ARLI_B_NOTIZ);
            fbController.AddParameter("@ARLI_A_BEZ1", artikel.ARLI_A_BEZ1);
            fbController.AddParameter("@ARLI_A_BEZ2", artikel.ARLI_A_BEZ2);
            fbController.AddParameter("@ARLI_A_BEZ3", artikel.ARLI_A_BEZ3);
            fbController.AddParameter("@ARLI_A_BEZ4", artikel.ARLI_A_BEZ4);
            fbController.AddParameter("@ARLI_A_BEZ5", artikel.ARLI_A_BEZ5);
            fbController.AddParameter("@ARLI_N_LASTUSER", artikel.ARLI_N_LASTUSER);
            fbController.AddParameter("@ARLI_TIMESTAMP", artikel.ARLI_TIMESTAMP);
            fbController.AddParameter("@WK5_ARLI_N_LIEFBESTAND", artikel.WK5_ARLI_N_LIEFBESTAND);
            fbController.AddParameter("@ARLI_L_NURVIELFACHES", artikel.ARLI_L_NURVIELFACHES);
            return fbController.QueryAsync(sql);
        }

        public static async Task UpdateOrCreateLieferantenArtikelAsync(Lieferantenartikel artikel, FbController2 fbController)
        {
            Lieferantenartikel? tmpArti = await GetLieferantenartikelAsync(artikel.ARLI_A_ARTIKELNR, artikel.ARLI_N_LIEFNR, fbController);

            if (tmpArti is not null)
            {
                await UpdateLieferantenArtikelAsync(artikel, fbController);
            }
            else
            {
                await CreateLieferantenArtikelAsync(artikel, fbController);
            }

        }

        public static async Task DeleteLieferantenArtikel(string ARLI_A_ARTIKELNR, int ARLI_N_LIEFNR)
        {
            using FbController2 fbController = new FbController2();

            await DeleteLieferantenArtikel(ARLI_A_ARTIKELNR, ARLI_N_LIEFNR, fbController);
        }

        public static async Task DeleteLieferantenArtikel(string ARLI_A_ARTIKELNR, int ARLI_N_LIEFNR, FbController2 fbController)
        {
            string sql = "DELETE FROM LIEFERANTENARTIKEL WHERE ARLI_A_ARTIKELNR = @ARLI_A_ARTIKELNR AND ARLI_N_LIEFNR = @ARLI_N_LIEFNR";

            fbController.AddParameter("@ARLI_A_ARTIKELNR", ARLI_A_ARTIKELNR);
            fbController.AddParameter("@ARLI_N_LIEFNR", ARLI_N_LIEFNR);
            await fbController.QueryAsync(sql);
        }
    }
}
