﻿using System;
using System.Threading.Tasks;
using WK5.Core.Services;

namespace WK5.Core.Models.Lager
{
    public class PositionBuchenInput
    {
        private decimal _menge;
        private decimal _preis;
        public int Auftragsnummer { get; set; }
        public int Chargennummer { get; set; }
        /// <summary>
        /// Ruft die zubuchende Artikelnummer ab.
        /// </summary>
        public string Artikelnummer { get; set; } = String.Empty;
        /// <summary>
        /// Ruft die Menge ab, die gebucht werden soll, oder legt diese fest.
        /// <para>Wenn versucht wird ein Wert <= 0 zu schreiben, dann wird die Menge automatisch auf 1 festgelegt.</para>
        /// </summary>
        public decimal Menge
        {
            get => _menge;
            set
            {
                _menge = value;

                if (_menge <= 0)
                {
                    _menge = 1;
                }


                if (SeriennummerBenötigt)
                {
                    ZugangSeriennummerInput[] tmp = Seriennummern;

                    Seriennummern = new ZugangSeriennummerInput[AnzahlBenötigterSeriennummern];

                    // Wir haben bereits Seriennummern eingegeben, die übernehmen wir nun, wir eine Menge eingeben.
                    for (int i = 0; i < tmp.Length && i < Seriennummern.Length; i++)
                    {
                        Seriennummern[i] = tmp[i];
                    }

                    for (int i = tmp.Length; i < Seriennummern.Length; i++)
                    {
                        Seriennummern[i] = new ZugangSeriennummerInput();
                    }
                }

            }
        }

        /// <summary>
        /// Ruft einen Wert ab, der angibt wie viele Seriennummern für die Buchung dieser Position benötigt werden.
        /// </summary>
        public int AnzahlBenötigterSeriennummern
        {
            get
            {
                if (!SeriennummerBenötigt)
                {
                    return 0;
                }

                return Convert.ToInt32(_menge);
            }
        }
        /// <summary>
        /// Ruft die Nummer der Bestellung ab. Dieser Wert ist nur dann gesetzt, wenn eine Position aus einer Bestellung übernommen wurde.
        /// </summary>
        public int Bestellnummer { get; set; } // Brauchen wir um zu Identifizieren aus welcher Bestellung
        /// <summary>
        /// Ruft die PositionsId innerhalbt der Bestellung ab. Dieser Wert ist nur dann gesetzt, wenn eine Position aus einer Bestellung übernommen wurde.
        /// </summary>
        public int BestellungPosId { get; set; }
        public ZugangSeriennummerInput[] Seriennummern { get; private set; } = Array.Empty<ZugangSeriennummerInput>();
        /// <summary>
        /// Ruft einen Wert ab, der angibt, ob für die Buchung dieses Artikels eine Seriennummer benötigt wird oder nicht.
        /// </summary>
        public bool SeriennummerBenötigt { get; private set; }

        /// <summary>
        /// Ruft einen Wert ab, der angibt, in welcher Menge der Artikel in der Bestellung noch nicht geliefert worden ist.
        /// <para>
        /// Wird die Position von einem Artikel, anstatt einer Bestellung übernommen, dann gibt dieser Wert diie Anzahl von offenen Bestellungen zurück.
        /// </para>
        /// </summary>
        public decimal Offen { get; }

        public string ChargenNotiz { get; set; } = String.Empty;
        public decimal Preis { get => Math.Round(_preis, 2); set => _preis = value; }
        public decimal Rabatt { get; set; }
        public decimal PreisGesamt => Preis * Menge;

        public string Bezeichnung { get; set; } = String.Empty;
        public string ArtikelnummerLieferant { get; set; } = String.Empty;

        public decimal BezahlterBedarf { get; set; }
        public decimal MwstSatz { get; set; }

        public bool Gebrauchtware { get; set; }
        public bool Geprüft { get; internal set; }

        public string RowClass => Geprüft ? "table-success" : String.Empty;
        public PositionBuchenInput(bool seriennummerBenötigt, decimal offen)
        {
            SeriennummerBenötigt = seriennummerBenötigt;
            Offen = offen;
        }


        public async Task RefreshCondtionsAsync()
        {
            using FbController2 fbController = new FbController2();
            ArtikelService artikelService = new ArtikelService();
            Artikel? artikel = await artikelService.GetAsync(Artikelnummer, fbController);

            if (artikel is not null)
            {
                SeriennummerBenötigt = artikel.ARTI_L_SN;
                if (SeriennummerBenötigt)
                {
                    // Refresh der Seriennummer Inputs
                    Menge = Menge;
                }
                else
                {
                    Seriennummern = Array.Empty<ZugangSeriennummerInput>();
                }

            }
        }
    }
}
