﻿using KarleyLibrary.Erweiterungen;
using Microsoft.AspNetCore.Components.Forms;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace WK5.Core.Models
{
    public static class Dokumentenablage
    {
        public static IEnumerable<Ablagedokument> GetDokumentenablage(DokumentenablageArt art, string zuordnung, CancellationToken cancellationToken)
        {
            if (!cancellationToken.IsCancellationRequested)
            {
                string ablagePfad = Path.Combine(GetBasisPfad(art), zuordnung);

                CheckDirectory(ablagePfad);

                DirectoryInfo di = new DirectoryInfo(ablagePfad);
                foreach (var file in di.GetFiles())
                {

                    if (cancellationToken.IsCancellationRequested)
                    {
                        break;
                    }

                    yield return new Ablagedokument(file);
                }
            }
        }

        public static async Task<FileInfo> UploadFileAsync(DokumentenablageArt art, string zuordnung, IBrowserFile upload)
        {
            string filename = Path.Combine(GetBasisPfad(art), zuordnung, upload.Name);
    
            await using FileStream fs = new(filename, FileMode.Create);
            await upload.OpenReadStream(upload.Size).CopyToAsync(fs);

            return new FileInfo(filename);
        }

        public static async Task<FileInfo> UploadPackbildAsync(DokumentenablageArt art, string zuordnung, IBrowserFile upload)
        {
            string full_filename = Path.Combine(GetBasisPfad(art), zuordnung, $"Packbild{Path.GetExtension(upload.Name)}");
            CheckDirectory(Path.Combine(GetBasisPfad(art), zuordnung));
            var (success, file, message) = FileErweiterung.FindSuitableFileName(full_filename);
            if(success)
            {
                await using FileStream fs = new(file, FileMode.Create);
                await upload.OpenReadStream(upload.Size).CopyToAsync(fs);

                return new FileInfo(full_filename);
            }
            else
            {
                throw new InvalidOperationException(message);
            }
            
        }
        private static string GetBasisPfad(DokumentenablageArt art) => art switch
        {
            DokumentenablageArt.Kunden => GlobalConfig.Configuration.KundenDokumentenAblagePfad,
            DokumentenablageArt.Lieferanten => GlobalConfig.Configuration.LieferantenDokumentenAblagePfad,
            DokumentenablageArt.Artikel => GlobalConfig.Configuration.ArtikelDokumentenAblagePfad,
            DokumentenablageArt.Angebote => GlobalConfig.Configuration.AngebotDokumentenAblagePfad,
            DokumentenablageArt.Aufträge => GlobalConfig.Configuration.AuftragDokumentenAblagePfad,
            DokumentenablageArt.Lieferscheine => GlobalConfig.Configuration.LieferscheinDokumentenAblagePfad,
            DokumentenablageArt.Rechnungen => GlobalConfig.Configuration.RechnungDokumentenAblagePfad,
            DokumentenablageArt.Gutschriften => GlobalConfig.Configuration.GutschriftDokumentenAblagePfad,
            DokumentenablageArt.RMA => GlobalConfig.Configuration.RmaDokumentenAblagePfad,
            DokumentenablageArt.Bestellungen => GlobalConfig.Configuration.BestellungenDokumentenAblagePfad,
            _ => throw new InvalidOperationException()
        };

        private static void CheckDirectory(string pfad)
        {
            if (!Directory.Exists(pfad))
            {
                Directory.CreateDirectory(pfad);
            }
        }
    }

    public class Ablagedokument
    {
        public FileInfo FileInfo { get; set; }
        public bool IsDownloading { get; set; }

        public string ViewUrl
        {
            get
            {
                string filename = FileInfo.FullName;
                filename = filename.Replace(GlobalConfig.Configuration.OutputPfad, string.Empty);

                return $"{GlobalConfig.Configuration.OutputUrl}/{filename}";
            }
        }
        public Ablagedokument(FileInfo fileInfo)
        {
            FileInfo = fileInfo;
        }
    }

    public enum DokumentenablageArt
    {
        Kunden,
        Lieferanten,
        Artikel,
        Angebote,
        Aufträge,
        Lieferscheine,
        Rechnungen,
        Gutschriften,
        RMA,
        Bestellungen
    }
}
