﻿using DvApi32;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace WK5.Core.Email.Provider
{
    /// <summary>
    /// Der Email Provider für den David Email Versand
    /// </summary>
    public class DavidProvider : IMailProvider
    {
        /// <summary>
        /// Die Anmeldedaten mit denen sich in der David API angemeledet werden kann. Dabei werden PORT und SSL ignoriert
        /// </summary>
        public MailCredentials Credentials { get; set; }
        /// <summary>
        /// Erstellt eine neue Instanz des <see cref="DavidProviders"/> mit den gegebenen Credentials
        /// </summary>
        /// <param name="credentials">Die zu nutzenden Credentials um sich bei der Tobit David API anzumelden</param>
        public DavidProvider(MailCredentials credentials)
        {
            Credentials = credentials;
        }

        /// <summary>
        /// Erstellt ein Objekt des Tobit Account mit den aktuellen Anmeldedaten
        /// </summary>
        /// <returns>Gibt ein <see cref="Account"/> Objekt zurück welches eine offene Verbindung zur Tobit API hält</returns>
        public Account GetDavidAccount()
        {
            DavidAPI oApi = new DavidAPI
            {
                LoginOptions = DvLoginOptions.DvLoginForceAsyncDuplicate
            };
            return oApi.Logon(Credentials.Host, Credentials.User, Credentials.Password, "", "", "AUTH");
        }

        /// <summary>
        /// Sendet eine Email über die Tobit API raus
        /// </summary>
        /// <param name="absender">Der Absender von dem die Email gesendet werden soll. Hier wird nur die Email Adresse beachtet</param>
        /// <param name="empfänger">Eine Liste aller Empfänger an die die Email gehen soll</param>
        /// <param name="betreff">Der Betreff der Email</param>
        /// <param name="body">Der Inhalt der Email</param>
        /// <param name="html">Gibt an ob die Email HTML benutzen soll oder nicht</param>
        /// <param name="anhänge">Eine Liste der zu versendenden Anhänge</param>
        /// <param name="bcc">Eine Liste aus Email Adressen die im Blind Carbon Copy stehen sollen</param>
        /// <param name="cc">Eine Liste aus Email Adressen die im Carbon Copy stehen sollen</param>
        /// <returns>Gibt eine <see cref="EmailResponse"/> zurück die indiziert ob der Versand erfolgreich war</returns>
        public EmailResponse Senden(EmailAbsender absender, List<string> empfänger, string betreff, string body, bool html = true, List<EmailAnhang>? anhänge = null, List<string>? bcc = null, List<string>? cc = null)
        {
            try
            {                
                Account oAccount = GetDavidAccount();
                Archive oArchive = oAccount.GetArchive(@"\\srv01\david\archive\zzzausgang");

                MailItem oMailItem = (MailItem)oArchive.NewItem(DvItemTypes.DvEMailItem);

                foreach (string adresse in empfänger)
                {
                    oMailItem.Recipients.Add(adresse, "", "");
                }

                if (bcc is not null)
                {
                    foreach (string bccMail in bcc)
                    {
                        oMailItem.BCC.Add(bccMail, "", "");
                    }
                }

                if (cc is not null)
                {
                    foreach (string ccMail in cc)
                    {
                        oMailItem.CC.Add(ccMail, "", "");
                    }
                }

                oMailItem.Subject = betreff;

                oMailItem.BodyText.Charset = "UTF-8";
                if (html)
                {
                    oMailItem.BodyText.HTMLText = body + $"@@von {absender.Email}@@";
                }
                else
                {
                    oMailItem.BodyText.PlainText = body + $"@@von {absender.Email}@@";
                }

                if (anhänge is not null)
                {

                    foreach (EmailAnhang anhang in anhänge)
                    {
                        oMailItem.Attachments.Add(anhang.Pfad, anhang.Name);
                    }
                }

                oMailItem.Send();

                return new EmailResponse(true);
            }
            catch (Exception ex)
            {
                return new EmailResponse(false, $"#EMAIL4 - David: {ex}");
            }
        }
    }
}
