﻿#define CREATE_PREVIEW_HTML
using iText.Html2pdf;
using KarleyLibrary.Erweiterungen;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using WK5.Core.Models;

namespace WK5.Core.Drucken.Aufträge
{
    public class PrintAuftrag : PrintBase
    {
        private readonly Auftrag _auftrag;
        private readonly PrintAuftragRegelsatz _regeln;
        private readonly OptionCollection _optionen;
        private readonly List<Lieferbedingung> _lieferbedingungen;
        private readonly ZahlungsbedingungCollection _zahlungsbedingungen;
        private readonly Mehrwertsteuer _versandMehrwertsteuer;


        public bool IsEnglisch { get; }
        private PrintAuftrag(Auftrag auftrag, PrintAuftragRegelsatz regeln, OptionCollection optionen, List<Lieferbedingung> lieferbedingungen, ZahlungsbedingungCollection zahlungsbedingungen, Mehrwertsteuer versandMehrwertsteuer)
        {
            _auftrag = auftrag;
            IsEnglisch = !GlobalConfig.DachLächer.Contains(_auftrag.Land);
            _regeln = regeln;
            _optionen = optionen;
            _lieferbedingungen = lieferbedingungen;
            _zahlungsbedingungen = zahlungsbedingungen;
            _versandMehrwertsteuer = versandMehrwertsteuer;
            _layoutFile = IsEnglisch ? "AuftragEnglisch.html" : "Auftrag.html";
            _template = String.Empty;
        }
        public static async Task<PrintAuftrag> CreateAsync(Auftrag auftrag, PrintAuftragRegelsatz regeln, FbController2 fbController)
        {
            var myClass = new PrintAuftrag
            (
                auftrag,
                regeln,
                await Option.GetOptionenAsync(fbController),
                await Lieferbedingung.GetLieferbedingungenAsync().ToListAsync(),
                await Zahlungsbedingung.GetZahlungsbedingungenAsync(fbController),
                await Mehrwertsteuer.GetVersandMehrwertsteuerAsync(fbController) ?? throw new ArgumentNullException(nameof(_versandMehrwertsteuer))
            );
            await myClass.InitializeAsync();
            return myClass;
        }
        public override string Print(string speicherpfad)
        {
            string filename = $"AU_{_auftrag.Belegnummer}.pdf";

#if CREATE_PREVIEW_HTML
            File.WriteAllText(Path.Combine(speicherpfad, $"AU_{_auftrag.Belegnummer}.html"), _template);
#endif
            byte[] byteArray = Encoding.UTF8.GetBytes(_template);
            MemoryStream stream = new MemoryStream(byteArray);
            ConverterProperties converterProperties = new ConverterProperties()
                .SetBaseUri(Path.Combine(AppDomain.CurrentDomain.BaseDirectory, LAYOUT_PATH));

            string fullPath = Path.Combine(speicherpfad, filename);
            if (File.Exists(fullPath))
            {
                File.Delete(fullPath);
            }

            HtmlConverter.ConvertToPdf(stream, new FileStream(fullPath, FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None), converterProperties);

            return fullPath;
        }

        protected override string GenerateDisplayCss()
        {
            StringBuilder cssBuilder = new StringBuilder();
            cssBuilder.AppendLine("<style>");



            if (!_regeln.ShowFooter)
            {
                cssBuilder.AppendLine("\t#footer {");
                //cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t\tvisibility: hidden;");
                cssBuilder.AppendLine("\t\topacity: 0;");
                cssBuilder.AppendLine("\t\tcolor: rgba(0,0,0,0);");
                cssBuilder.AppendLine("\t\ttransform: scale(0);");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowHeader)
            {
                cssBuilder.AppendLine("\t.karley-header {");
                //cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t\tvisibility: hidden;");
                cssBuilder.AppendLine("\t\topacity: 0;");
                cssBuilder.AppendLine("\t\tcolor: rgba(0,0,0,0);");
                cssBuilder.AppendLine("\t\ttransform: scale(0);");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowBestelltDurch)
            {
                cssBuilder.AppendLine("\t.display-bestellt-durch{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowBestelltAm)
            {
                cssBuilder.AppendLine("\t.display-bestellt-am{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowRechnungsanschrift)
            {
                cssBuilder.AppendLine("\t.display-rechnungsanschrift{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowLieferanschrift)
            {
                cssBuilder.AppendLine("\t.display-lieferanschrift{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowLieferzeit)
            {
                cssBuilder.AppendLine("\t.display-lieferzeit{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowMwstUndBrutto)
            {
                cssBuilder.AppendLine("\t.display-mwst, .display-endbetrag-brutto{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowZwischensumme)
            {
                cssBuilder.AppendLine("\t.display-zwischensumme{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowRabatt)
            {
                cssBuilder.AppendLine("\t.display-rabatt{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowBestellnummer)
            {
                cssBuilder.AppendLine("\t.display-bestellnummer{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowVerpackungskosten)
            {
                cssBuilder.AppendLine("\t.display-verpackungskosten{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowVersicherungskosten)
            {
                cssBuilder.AppendLine("\t.display-versicherungskosten{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowVersandkosten)
            {
                cssBuilder.AppendLine("\t.display-versandkosten{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }



            if (!_regeln.ShowAdresszeileLand)
            {
                cssBuilder.AppendLine("\t.display-empfaenger-land{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAdresszeile2)
            {
                cssBuilder.AppendLine("\t.display-addresszeile2{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAdresszeile3)
            {
                cssBuilder.AppendLine("\t.display-addresszeile3{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAdresszeile4)
            {
                cssBuilder.AppendLine("\t.display-addresszeile4{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAngebotsnummer)
            {
                cssBuilder.AppendLine("\t.display-angebotsnummer{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowSpeditionAuftragsnummer)
            {
                cssBuilder.AppendLine("\t.display-speditionauftragsnummer{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }





            cssBuilder.AppendLine("</style>");

            return cssBuilder.ToString();
        }

        protected override async Task InitializeAsync()
        {
            string layoutPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, LAYOUT_PATH, _layoutFile);
            _template = await File.ReadAllTextAsync(layoutPath);
            SetRegeln();
            _template = await SetTemplateVariables();
        }

        protected override Task InitializeAsync(FbController2 fbController)
        {
            throw new NotImplementedException();
        }

        protected override void SetRegeln()
        {
            _regeln.ShowLieferzeit = !_auftrag.Lieferzeit.Equals(String.Empty);
            _regeln.ShowMwstUndBrutto = _auftrag.MwstBerechnen;
            _regeln.ShowRabatt = _auftrag.Positionen.Where(x => x.GetRabatt1() > 0 || x.GetRabatt2() > 0).Any();
            _regeln.ShowBestelltDurch = !String.IsNullOrWhiteSpace(_auftrag.BestelltDurch);
            _regeln.ShowBestelltAm = _auftrag.BestelltAm != default;
            _regeln.ShowLieferanschrift = _auftrag.LieferanschriftId > 0;
            _regeln.ShowRechnungsanschrift = _auftrag.RechnungsanschriftId > 0;
            _regeln.ShowBestellnummer = !String.IsNullOrWhiteSpace(_auftrag.Bestellnummer);
            _regeln.ShowVerpackungskosten = _auftrag.Verpackungskosten > 0;
            _regeln.ShowVersandkosten = _auftrag.Frachtkosten > 0;
            _regeln.ShowVersicherungskosten = _auftrag.Versicherungskosten > 0;
            _regeln.ShowZwischensumme = _regeln.ShowVersicherungskosten || _regeln.ShowVersandkosten || _regeln.ShowVerpackungskosten || _regeln.ShowRabatt;
            _regeln.ShowAdresszeileLand = _auftrag.Land != "DE";
            _regeln.ShowAdresszeile2 = !String.IsNullOrWhiteSpace(_auftrag.Name2);
            _regeln.ShowAdresszeile3 = !String.IsNullOrWhiteSpace(_auftrag.Name3);
            _regeln.ShowAdresszeile4 = !String.IsNullOrWhiteSpace(_auftrag.Ansprechpartner);
            _regeln.ShowAngebotsnummer = _auftrag.AngebotsnummerVerknüpfung > 0;
            _regeln.ShowSpeditionAuftragsnummer = !string.IsNullOrEmpty(_auftrag.SpeditionsAuftragsnummer);
        }

        protected override async Task<string> SetTemplateVariables()
        {
            Ansprechpartner? partner = null;
            if (!String.IsNullOrWhiteSpace(_auftrag.Ansprechpartner))
            {
                partner = await Ansprechpartner.GetAnsprechpartnerByNameAsync(_auftrag.Kundennummer, _auftrag.Ansprechpartner);
            }

            StringBuilder posBuilder = new StringBuilder();
            decimal netto = 0;
            decimal rabatt = 0;
            decimal mwst = 0;
            int zähler = 1;
            foreach (var pos in _auftrag.Positionen.Where(x => x.IstBundle || (!x.IstBundle && x.BundleId == 0)).OrderBy(x => x.PosNr))
            {
                if (pos.PosOhneDruck)
                {
                    continue;
                }

                bool istText = pos.Artikelnummer.Equals("TEXT", StringComparison.OrdinalIgnoreCase);

                Option? sucheOption = _optionen.Where(x => x.OPTI_A_NR.Equals(pos.Option)).FirstOrDefault();

                posBuilder.AppendLine("<tr class=\"pos\">");
                posBuilder.AppendLine("\t<td class=\"nowrap\">");
                if (!istText)
                {
                    posBuilder.AppendLine($"\t\t{zähler}");
                    zähler++;
                }
                posBuilder.AppendLine("\t</td>");
                // Spalte Artikelnummer
                posBuilder.AppendLine("\t<td class=\"\">");
                if (!istText)
                {
                    posBuilder.AppendLine("\t\t<span class=\"font-weight-bold\">");
                    if (!String.IsNullOrWhiteSpace(pos.Kundensachnummer))
                    {
                        posBuilder.AppendLine($"\t\t\t{pos.Artikelnummer} ({pos.Kundensachnummer})");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t\t{pos.Artikelnummer}");
                    }
                    posBuilder.AppendLine("\t\t</span>");
                }

                if (pos.ARTI_L_MIETE && pos.BPOS_D_MIETE_START != default && pos.BPOS_D_MIETE_ENDE != default)
                {
                    if (IsEnglisch)
                    {
                        posBuilder.AppendLine($"\t\t<span class=\"d-block font-weight-bold\">Period ahead.: {pos.BPOS_D_MIETE_START.ToShortDateString()} - {pos.BPOS_D_MIETE_ENDE.ToShortDateString()}</span>");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t<span class=\"d-block font-weight-bold\">Zeitraum voraus.: {pos.BPOS_D_MIETE_START.ToShortDateString()} - {pos.BPOS_D_MIETE_ENDE.ToShortDateString()}</span>");
                    }
                }

                posBuilder.AppendLine($"\t\t<div class=\"position-bezeichnung\">");
                posBuilder.AppendLine($"\t\t\t{pos.GetBezeichnung()}");

                if (pos.IstBundle)
                {
                    if (IsEnglisch)
                    {
                        posBuilder.AppendLine($"\t\t\t<span class=\"d-block font-weight-bold bundle-heading\">Bundle consists of:</span>");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t\t<span class=\"d-block font-weight-bold bundle-heading\">Bundle bestehend aus:</span>");
                    }

                    foreach (var bundlePos in pos.BundleArtikel)
                    {
                        decimal basisBundleMenge = bundlePos.Menge / pos.Menge;
                        posBuilder.AppendLine($"\t\t\t<div class=\"bundle-pos\">");
                        posBuilder.AppendLine($"\t\t\t\t<span class=\"bundle-pos-quantity\">{basisBundleMenge} x {bundlePos.Artikelnummer}</span>");
                        posBuilder.AppendLine($"\t\t\t\t<span class=\"bundle-pos-description\">{bundlePos.Bezeichnung1}</span>");
                        posBuilder.AppendLine($"\t\t\t</div>");
                    }
                }

                posBuilder.AppendLine($"\t\t</div>");
                posBuilder.AppendLine("\t</td>");

                // Spalte Menge
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (pos.Menge > 0)
                {
                    posBuilder.AppendLine($"\t\t{pos.Menge:0.00} {pos.BPOS_A_MASSEINHEIT}");
                }
                else
                {
                    posBuilder.AppendLine($"\t\t");
                }

                // Spalte VK-Preis
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (!istText)
                {
                    if (pos.Menge > 10000)
                    {
                        posBuilder.AppendLine($"\t\t{pos.Preis:0.0000}");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t{pos.Preis:0.00}");
                    }

                }
                posBuilder.AppendLine("\t</td>");

                // Spalte Rabatt
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (pos.Rabatt1 > 0 && pos.Rabatt2 > 0)
                {
                    if (!String.IsNullOrWhiteSpace(pos.Rabattbezeichnung))
                    {
                        if (IsEnglisch)
                        {
                            // Bei Englisch gibt es keine Rabattbezeichnung, da die nur auf deutsch vorhanden ist.
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />+ {pos.Rabatt2}<br />Special discount");
                        }
                        else
                        {
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />{pos.Rabattbezeichnung}<br />+ {pos.Rabatt2:0.00}<br />Sonderrabatt");
                        }

                    }
                    else
                    {
                        if (IsEnglisch)
                        {
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />+ {pos.Rabatt2:0.00}<br />Special discount");
                        }
                        else
                        {
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />+ {pos.Rabatt2:0.00}<br />Sonderrabatt");
                        }
                    }

                }
                else if (pos.Rabatt1 > 0)
                {

                    if (!String.IsNullOrWhiteSpace(pos.Rabattbezeichnung))
                    {
                        posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />");
                        posBuilder.AppendLine($"\t\t{pos.Rabattbezeichnung}");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}");
                    }
                }
                else if (pos.Rabatt2 > 0)
                {
                    posBuilder.AppendLine($"\t\t{pos.Rabatt2:0.00}<br />");
                    if (IsEnglisch)
                    {
                        posBuilder.AppendLine("\t\tSpecial discount");
                    }
                    else
                    {
                        posBuilder.AppendLine("\t\tSonderrabatt");
                    }
                }
                posBuilder.AppendLine("\t</td>");

                // Spalte Preis
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (pos.PreisMitRabatt > 0 || pos.Rabatt > 0)
                {
                    posBuilder.AppendLine($"\t\t{pos.PreisMitRabatt:0.00}");
                }
                posBuilder.AppendLine("\t</td>");

                // Spalte Gesamt
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (!istText)
                {
                    if (sucheOption is not null && !sucheOption.OPTI_L_BERECHNEN)
                    {
                        posBuilder.AppendLine($"\t\t{pos.GetGesamtNetto():0.00}<br /><span class=\"font-weight-bold\">{sucheOption.OPTI_A_BEZEICH}<br />nicht in Summe</span>");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t{(pos.GetGesamtNetto() - pos.Rabatt):0.00}");
                    }
                }
                posBuilder.AppendLine("\t</td>");



                posBuilder.AppendLine("\t</td>");
                posBuilder.AppendLine("</tr>");


                if (!String.IsNullOrWhiteSpace(pos.Langtext))
                {
                    posBuilder.AppendLine("<tr>");
                    posBuilder.AppendLine("\t<td></td>");
                    posBuilder.AppendLine("\t<td colspan=\"5\">");
                    posBuilder.AppendLine($"\t\t<div class=\"position-bezeichnung\">");
                    if (pos.LangtextDrucken)
                    {
                        posBuilder.AppendLine($"\t\t\t\t{pos.Langtext.HtmlToDruckHtml()}");
                    }
                    posBuilder.AppendLine($"\t\t</div>");
                    posBuilder.AppendLine("\t</td>");
                    posBuilder.AppendLine("\t<td></td>");
                    posBuilder.AppendLine("</tr>");
                }


                if ((sucheOption is not null && sucheOption.OPTI_L_BERECHNEN) || sucheOption is null)
                {
                    mwst += pos.GetMwstSumme();
                    rabatt += pos.GetRabatt1();
                    rabatt += pos.GetRabatt2();
                    netto += pos.GetGesamtNetto();
                }
            }


            mwst += _auftrag.Zusatzkosten * _versandMehrwertsteuer.MWST_N_PROZENT / 100;

            if (_regeln.ShowZwischensumme)
            {
                posBuilder.AppendLine("<tr class=\"total display-zwischensumme border-top\">");
            }
            else
            {
                posBuilder.AppendLine("<tr class=\"total display-zwischensumme\">");
            }

            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tSubtotal:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tZwischensumme:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{(netto - rabatt):0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-versandkosten\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tShipping costs:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tVersandkosten:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{_auftrag.Frachtkosten:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-verpackungskosten\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tPacking costs:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tVerpackungskosten:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{_auftrag.Verpackungskosten:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-versicherungskosten\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tInsurance costs:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tVersicherungskosten:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{_auftrag.Versicherungskosten:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");



            if (_regeln.ShowZwischensumme)
            {
                posBuilder.AppendLine("<tr class=\"total font-weight-bold\">");
            }
            else
            {
                posBuilder.AppendLine("<tr class=\"total border-top font-weight-bold\">");
            }

            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tTotal EUR without VAT.:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tTotal EUR ohne Mwst.:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{netto + _auftrag.Zusatzkosten - rabatt:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-mwst\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            posBuilder.AppendLine($"\t\tzzgl. {_auftrag.BELE_N_MWST_SATZ1}% MwSt:");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{mwst:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-endbetrag-brutto font-weight-bold\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tTotal EUR incl. VAT.");
            }
            else
            {
                posBuilder.AppendLine($"\t\tTotal EUR inkl. Mwst.:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{netto + _auftrag.Zusatzkosten - rabatt + mwst:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-rabatt\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\t(Discount:");
            }
            else
            {
                posBuilder.AppendLine($"\t\t(eingerechneter Rabatt:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t-{rabatt:0.00})");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            Lieferanschrift? lieferanschrift = null;
            Rechnungsanschrift? rechnungsanschrift = null;
            Lieferbedingung? lieferbedingung = _lieferbedingungen.Where(x => x.LIBD_N_NR == _auftrag.LieferbedingungId).FirstOrDefault();
            Zahlungsbedingung? zahlungsbedingung = _zahlungsbedingungen.Where(x => x.ZABD_N_NR == _auftrag.ZahlungsbedingungId).FirstOrDefault();
            if (_auftrag.LieferanschriftId > 0)
            {
                lieferanschrift = await Lieferanschrift.GetLieferanschriftAsync(_auftrag.Kundennummer, _auftrag.LieferanschriftId);
            }

            if (_auftrag.RechnungsanschriftId > 0)
            {
                rechnungsanschrift = await Rechnungsanschrift.GetRechnungsanschriftAsync(_auftrag.Kundennummer, _auftrag.RechnungsanschriftId);
            }

            using FbController2 fbController = new FbController2();

            Land land = await Land.GetLandAsync(_auftrag.Land, fbController) ?? throw new ArgumentNullException(nameof(land));

            string neutraleLieferungText = String.Empty;

            if (_auftrag.NeutralerVersender)
            {
                if (IsEnglisch)
                {
                    neutraleLieferungText = $"<span class=\"d-block font-weight-bold h4\">XXX Caution - Neutral Delivery XXX</span>";
                }
                else
                {
                    neutraleLieferungText = $"<span class=\"d-block font-weight-bold h4\">XXX Vorsicht - Neutrale Lieferung XXX</span>";
                }
            }

            if (rabatt is 0)
            {
                _regeln.ShowRabatt = false;
            }

            _template = _template
                .Replace("{{ DISPLAY_CSS }}", GenerateDisplayCss())
                .Replace("{{ IMPRESSUM }}", GlobalConfig.Configuration.FirmenDaten.ImpressumHtmlDruckformulare())
                .Replace("{{ ABS_FIRMA }}", GlobalConfig.Configuration.FirmenDaten.Firmenname)
                .Replace("{{ ABS_STRASSE }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Strasse)
                .Replace("{{ ABS_LAND }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Land)
                .Replace("{{ ABS_PLZ }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Postleitzahl)
                .Replace("{{ ABS_ORT }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Ort)
                .Replace("{{ ADDRESSZEILE1 }}", _auftrag.Name1)
                .Replace("{{ ADDRESSZEILE2 }}", _auftrag.Name2)
                .Replace("{{ ADDRESSZEILE3 }}", _auftrag.Name3)
                .Replace("{{ ADDRESSZEILE4 }}", partner is not null ? partner.GetName() : _auftrag.Ansprechpartner)
                .Replace("{{ EMP_NAME }}", partner?.PART_A_NAME ?? String.Empty)
                .Replace("{{ EMP_STRASSE }}", _auftrag.Strasse)
                .Replace("{{ EMP_LAND }}", land.LAND_A_NAME)
                .Replace("{{ EMP_PLZ }}", _auftrag.Postleitzahl)
                .Replace("{{ EMP_ORT }}", _auftrag.Ort)
                .Replace("{{ DATUM }}", _auftrag.BELE_D_DATE.ToShortDateString())
                .Replace("{{ KUNDENNUMMER }}", _auftrag.Kundennummer)
                .Replace("{{ BESTELLT_DURCH }}", _auftrag.BestelltDurch)
                .Replace("{{ BESTELLT_AM }}", _auftrag.BestelltAm.ToShortDateString())
                .Replace("{{ KOPFTEXT }}", _auftrag.Kopftext)
                .Replace("{{ FUSSTEXT }}", _auftrag.Fußtext)
                .Replace("{{ POSITIONEN }}", posBuilder.ToString())
                .Replace("{{ AUFTRAGSNUMMER }}", _auftrag.Belegnummer.ToString())
                .Replace("{{ LIEFERANSCHRIFT }}", lieferanschrift?.ToString() ?? String.Empty)
                .Replace("{{ RECHNUNGSANSCHRIFT }}", rechnungsanschrift?.ToString() ?? String.Empty)
                .Replace("{{ LIEFERZEIT }}", _auftrag.Lieferzeit)
                .Replace("{{ LIEFERBEDINGUNGEN }}", IsEnglisch
                ? lieferbedingung?.LIBD_A_TEXT5 ?? String.Empty
                : lieferbedingung?.LIBD_A_TEXT1 ?? String.Empty)
                .Replace("{{ ZAHLUNGSBEDINGUNGEN }}", IsEnglisch
                ? zahlungsbedingung?.ZABD_A_TEXT2 ?? String.Empty
                : zahlungsbedingung?.ZABD_A_TEXT1 ?? String.Empty)
                .Replace("{{ BESTELLNUMMER }}", _auftrag.Bestellnummer ?? String.Empty)
                .Replace("{{ ANGEBOT_NR }}", _auftrag.AngebotsnummerVerknüpfung.ToString())
                .Replace("{{ NEUTRALE_LIEFERUNG }}", neutraleLieferungText)
                .Replace("{{ SPEDITIONSAUFTRAGSNUMMER }}", _auftrag.SpeditionsAuftragsnummer)
                ;

            return await Task.FromResult(_template);
        }
    }
}
