﻿#define CREATE_PREVIEW_HTML
using iText.Html2pdf;
using KarleyLibrary.Erweiterungen;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using WK5.Core.Models;

namespace WK5.Core.Drucken.Angebote
{
    public class PrintAngebot : PrintBase
    {
        private readonly Angebot _angebot;
        private readonly PrintAngebotRegelsatz _regeln;
        private readonly List<Option> _optionen;
        private readonly List<Lieferbedingung> _lieferbedingungen;
        private readonly ZahlungsbedingungCollection _zahlungsbedingungen;
        private readonly Mehrwertsteuer _versandMehrwertsteuer;

        public bool IsEnglisch { get; }
        private PrintAngebot(Angebot angebot, PrintAngebotRegelsatz regeln, List<Option> optionen, List<Lieferbedingung> lieferbedingungen, ZahlungsbedingungCollection zahlungsbedingungen, Mehrwertsteuer versandMehrwertsteuer)
        {
            _angebot = angebot;
            IsEnglisch = !GlobalConfig.DachLächer.Contains(_angebot.Land);
            _regeln = regeln;
            _optionen = optionen;
            _lieferbedingungen = lieferbedingungen;
            _zahlungsbedingungen = zahlungsbedingungen;
            _versandMehrwertsteuer = versandMehrwertsteuer;
            _layoutFile = IsEnglisch ? "AngebotEnglisch.html" : "Angebot.html";
            _template = String.Empty;
        }
        public static async Task<PrintAngebot> CreateAsync(Angebot angebot, PrintAngebotRegelsatz regeln, FbController2 fbController)
        {
            var myClass = new PrintAngebot
            (
                angebot,
                regeln,
                await Option.GetOptionenAsync(fbController),
                await Lieferbedingung.GetLieferbedingungenAsync().ToListAsync(),
                await Zahlungsbedingung.GetZahlungsbedingungenAsync(fbController),
                await Mehrwertsteuer.GetVersandMehrwertsteuerAsync(fbController) ?? throw new ArgumentNullException(nameof(_versandMehrwertsteuer))
            );
            await myClass.InitializeAsync();
            return myClass;
        }
        public override string Print(string speicherpfad)
        {
            string filename = $"AN_{_angebot.Belegnummer}.pdf";

#if CREATE_PREVIEW_HTML
            File.WriteAllText(Path.Combine(speicherpfad, $"AN_{ _angebot.Belegnummer}.html"), _template);
#endif

            byte[] byteArray = Encoding.UTF8.GetBytes(_template);
            MemoryStream stream = new MemoryStream(byteArray);
            ConverterProperties converterProperties = new ConverterProperties()
                .SetBaseUri(Path.Combine(AppDomain.CurrentDomain.BaseDirectory, LAYOUT_PATH));

            string fullPath = Path.Combine(speicherpfad, filename);
            if (File.Exists(fullPath))
            {
                File.Delete(fullPath);
            }

            HtmlConverter.ConvertToPdf(stream, new FileStream(fullPath, FileMode.OpenOrCreate, FileAccess.ReadWrite, FileShare.None), converterProperties);

            return fullPath;
        }        

        protected override string GenerateDisplayCss()
        {
            StringBuilder cssBuilder = new StringBuilder();
            cssBuilder.AppendLine("<style>");



            if (!_regeln.ShowFooter)
            {
                cssBuilder.AppendLine("\t#footer {");
                //cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t\tvisibility: hidden;");
                cssBuilder.AppendLine("\t\topacity: 0;");
                cssBuilder.AppendLine("\t\tcolor: rgba(0,0,0,0);");
                cssBuilder.AppendLine("\t\ttransform: scale(0);");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowHeader)
            {
                cssBuilder.AppendLine("\t.karley-header {");
                //cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t\tvisibility: hidden;");
                cssBuilder.AppendLine("\t\topacity: 0;");
                cssBuilder.AppendLine("\t\tcolor: rgba(0,0,0,0);");
                cssBuilder.AppendLine("\t\ttransform: scale(0);");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAnfrageDurch)
            {
                cssBuilder.AppendLine("\t.display-anfrage-durch{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAnfrageVom)
            {
                cssBuilder.AppendLine("\t.display-anfrage-vom{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowRechnungsanschrift)
            {
                cssBuilder.AppendLine("\t.display-rechnungsanschrift{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowLieferanschrift)
            {
                cssBuilder.AppendLine("\t.display-lieferanschrift{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowLieferzeit)
            {
                cssBuilder.AppendLine("\t.display-lieferzeit{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowMwstUndBrutto)
            {
                cssBuilder.AppendLine("\t.display-mwst, .display-endbetrag-brutto{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowZwischensumme)
            {
                cssBuilder.AppendLine("\t.display-zwischensumme{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowRabatt)
            {
                cssBuilder.AppendLine("\t.display-rabatt{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAnfragennummer)
            {
                cssBuilder.AppendLine("\t.display-anfragennummer{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowVerpackungskosten)
            {
                cssBuilder.AppendLine("\t.display-verpackungskosten{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowVersicherungskosten)
            {
                cssBuilder.AppendLine("\t.display-versicherungskosten{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowVersandkosten)
            {
                cssBuilder.AppendLine("\t.display-versandkosten{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }
            if (!_regeln.ShowAdresszeile2)
            {
                cssBuilder.AppendLine("\t.display-addresszeile2{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAdresszeile3)
            {
                cssBuilder.AppendLine("\t.display-addresszeile3{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAdresszeile4)
            {
                cssBuilder.AppendLine("\t.display-addresszeile4{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }

            if (!_regeln.ShowAdresszeileLand)
            {
                cssBuilder.AppendLine("\t.display-empfaenger-land{");
                cssBuilder.AppendLine("\t\tdisplay: none;");
                cssBuilder.AppendLine("\t}");
            }






            cssBuilder.AppendLine("</style>");

            return cssBuilder.ToString();
        }

        protected override async Task InitializeAsync()
        {
            string layoutPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, LAYOUT_PATH, _layoutFile);
            _template = await File.ReadAllTextAsync(layoutPath);
            SetRegeln();
            _template = await SetTemplateVariables();
        }

        protected override Task InitializeAsync(FbController2 fbController)
        {
            throw new NotImplementedException();
        }

        protected override void SetRegeln()
        {
            _regeln.ShowLieferzeit = !_angebot.Lieferzeit.Equals(String.Empty);
            _regeln.ShowMwstUndBrutto = _angebot.MwstBerechnen;
            _regeln.ShowRabatt = _angebot.Positionen.Where(x => x.GetRabatt1() > 0 || x.GetRabatt2() > 0).Any();
            _regeln.ShowAnfrageDurch = !String.IsNullOrWhiteSpace(_angebot.AnfrageDurch);
            _regeln.ShowAnfrageVom = _angebot.AnfrageVom != default;
            _regeln.ShowLieferanschrift = _angebot.LieferanschriftId > 0;
            _regeln.ShowRechnungsanschrift = _angebot.RechnungsanschriftId > 0;
            _regeln.ShowAnfragennummer = !String.IsNullOrWhiteSpace(_angebot.Anfragennummer);
            _regeln.ShowVerpackungskosten = _angebot.Verpackungskosten > 0;
            _regeln.ShowVersandkosten = _angebot.Frachtkosten > 0;
            _regeln.ShowVersicherungskosten = _angebot.Versicherungskosten > 0;
            _regeln.ShowZwischensumme = _regeln.ShowVersicherungskosten || _regeln.ShowVersandkosten || _regeln.ShowVerpackungskosten || _regeln.ShowRabatt;
            _regeln.ShowAdresszeileLand = _angebot.Land != "DE";
            _regeln.ShowAdresszeile2 = !String.IsNullOrWhiteSpace(_angebot.Name2);
            _regeln.ShowAdresszeile3 = !String.IsNullOrWhiteSpace(_angebot.Name3);
            _regeln.ShowAdresszeile4 = !String.IsNullOrWhiteSpace(_angebot.Ansprechpartner);
        }

        protected override async Task<string> SetTemplateVariables()
        {
            Ansprechpartner? partner = null;
            if (!String.IsNullOrWhiteSpace(_angebot.Ansprechpartner))
            {
                partner = await Ansprechpartner.GetAnsprechpartnerByNameAsync(_angebot.Kundennummer, _angebot.Anrede);
            }
            using FbController2 fbController = new FbController2();
            StringBuilder posBuilder = new StringBuilder();
            decimal netto = 0;
            decimal rabatt = 0;
            decimal mwst = 0;
            int zähler = 1;
            foreach (var pos in _angebot.Positionen.Where(x => x.IstBundle || (!x.IstBundle && x.BundleId == 0)).OrderBy(x => x.PosNr))
            {
                if (pos.PosOhneDruck)
                {
                    continue;
                }

                bool istText = pos.Artikelnummer.Equals("TEXT", StringComparison.OrdinalIgnoreCase);

                Option? sucheOption = _optionen.Where(x => x.OPTI_A_NR.Equals(pos.Option)).FirstOrDefault();

                posBuilder.AppendLine("<tr class=\"pos\">");
                posBuilder.AppendLine("\t<td class=\"nowrap\">");
                if (!istText)
                {
                    posBuilder.AppendLine($"\t\t{zähler}");
                    zähler++;
                }
                posBuilder.AppendLine("\t</td>");
                // Spalte Artikelnummer
                posBuilder.AppendLine("\t<td class=\"\">");
                if (!istText)
                {
                    posBuilder.AppendLine("\t\t<span class=\"font-weight-bold\">");
                    if (!String.IsNullOrWhiteSpace(pos.Kundensachnummer))
                    {
                        posBuilder.AppendLine($"\t\t\t{pos.Artikelnummer} ({pos.Kundensachnummer})");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t\t{pos.Artikelnummer}");
                    }
                    posBuilder.AppendLine("\t\t</span>");
                }

                if (pos.ARTI_L_MIETE && pos.BPOS_D_MIETE_START != default && pos.BPOS_D_MIETE_ENDE != default)
                {
                    if (IsEnglisch)
                    {
                        posBuilder.AppendLine($"\t\t<span class=\"d-block font-weight-bold\">Period ahead.: {pos.BPOS_D_MIETE_START.ToShortDateString()} - {pos.BPOS_D_MIETE_ENDE.ToShortDateString()}</span>");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t<span class=\"d-block font-weight-bold\">Zeitraum voraus.: {pos.BPOS_D_MIETE_START.ToShortDateString()} - {pos.BPOS_D_MIETE_ENDE.ToShortDateString()}</span>");
                    }
                }

                posBuilder.AppendLine($"\t\t<div class=\"position-bezeichnung\">");
                posBuilder.AppendLine($"\t\t\t{pos.GetBezeichnung()}");
                if (pos.IstBundle)
                {
                    if (IsEnglisch)
                    {
                        posBuilder.AppendLine($"\t\t\t<span class=\"d-block font-weight-bold bundle-heading\">Bundle consists of:</span>");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t\t<span class=\"d-block font-weight-bold bundle-heading\">Bundle bestehend aus:</span>");
                    }

                    foreach (var bundlePos in pos.BundleArtikel)
                    {
                        decimal basisBundleMenge = bundlePos.Menge / pos.Menge;
                        posBuilder.AppendLine($"\t\t\t<div class=\"bundle-pos\">");
                        posBuilder.AppendLine($"\t\t\t\t<span class=\"bundle-pos-quantity\">{basisBundleMenge} x {bundlePos.Artikelnummer}</span>");
                        posBuilder.AppendLine($"\t\t\t\t<span class=\"bundle-pos-description\">{bundlePos.Bezeichnung1}</span>");
                        posBuilder.AppendLine($"\t\t\t</div>");
                    }
                }
                posBuilder.AppendLine($"\t\t</div>");
                posBuilder.AppendLine("\t</td>");

                // Spalte Menge
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (pos.Menge > 0)
                {
                    posBuilder.AppendLine($"\t\t{pos.Menge:0.00} {pos.BPOS_A_MASSEINHEIT}");
                }
                else
                {
                    posBuilder.AppendLine($"\t\t");
                }

                // Spalte VK-Preis
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (!istText)
                {
                    if (pos.Menge > 10000)
                    {
                        posBuilder.AppendLine($"\t\t{pos.Preis:0.0000}");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t{pos.Preis:0.00}");
                    }

                }
                posBuilder.AppendLine("\t</td>");

                // Spalte Rabatt
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (pos.Rabatt1 > 0 && pos.Rabatt2 > 0)
                {
                    if (!String.IsNullOrWhiteSpace(pos.Rabattbezeichnung))
                    {
                        if (IsEnglisch)
                        {
                            // Bei Englisch gibt es keine Rabattbezeichnung, da die nur auf deutsch vorhanden ist.
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />+ {pos.Rabatt2}<br />Special discount");
                        }
                        else
                        {
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />{pos.Rabattbezeichnung}<br />+ {pos.Rabatt2:0.00}<br />Sonderrabatt");
                        }

                    }
                    else
                    {
                        if (IsEnglisch)
                        {
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />+ {pos.Rabatt2:0.00}<br />Special discount");
                        }
                        else
                        {
                            posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />+ {pos.Rabatt2:0.00}<br />Sonderrabatt");
                        }
                    }

                }
                else if (pos.Rabatt1 > 0)
                {

                    if (!String.IsNullOrWhiteSpace(pos.Rabattbezeichnung))
                    {
                        posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}<br />");
                        posBuilder.AppendLine($"\t\t{pos.Rabattbezeichnung}");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t{pos.Rabatt1:0.00}");
                    }
                }
                else if (pos.Rabatt2 > 0)
                {
                    posBuilder.AppendLine($"\t\t{pos.Rabatt2:0.00}<br />");
                    if (IsEnglisch)
                    {
                        posBuilder.AppendLine("\t\tSpecial discount");
                    }
                    else
                    {
                        posBuilder.AppendLine("\t\tSonderrabatt");
                    }
                }
                posBuilder.AppendLine("\t</td>");

                // Spalte Preis
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (pos.PreisMitRabatt > 0 || pos.Rabatt > 0)
                {
                    posBuilder.AppendLine($"\t\t{pos.PreisMitRabatt:0.00}");
                }
                posBuilder.AppendLine("\t</td>");

                // Spalte Gesamt
                posBuilder.AppendLine("\t<td class=\"nowrap text-right\">");
                if (!istText)
                {
                    if (sucheOption is not null && !sucheOption.OPTI_L_BERECHNEN)
                    {
                        posBuilder.AppendLine($"\t\t{pos.GetGesamtNetto():0.00}<br /><span class=\"font-weight-bold\">{sucheOption.OPTI_A_BEZEICH}<br />nicht in Summe</span>");
                    }
                    else
                    {
                        posBuilder.AppendLine($"\t\t{(pos.GetGesamtNetto() - pos.Rabatt):0.00}");
                    }
                }
                posBuilder.AppendLine("\t</td>");



                posBuilder.AppendLine("\t</td>");
                posBuilder.AppendLine("</tr>");

                if (!String.IsNullOrWhiteSpace(pos.Langtext))
                {
                    posBuilder.AppendLine("<tr>");
                    posBuilder.AppendLine("\t<td></td>");
                    posBuilder.AppendLine("\t<td colspan=\"5\">");
                    posBuilder.AppendLine($"\t\t<div class=\"position-bezeichnung\">");
                    if (pos.LangtextDrucken)
                    {
                        posBuilder.AppendLine($"\t\t\t\t{pos.Langtext.HtmlToDruckHtml()}");
                    }
                    posBuilder.AppendLine($"\t\t</div>");
                    posBuilder.AppendLine("\t</td>");
                    posBuilder.AppendLine("\t<td></td>");
                    posBuilder.AppendLine("</tr>");
                }

                if (!String.IsNullOrWhiteSpace(pos.Bild))
                {
                    ArtikelBild? bild = await ArtikelBild.GetArtikelbildByIdAsync(pos.BildId, fbController);

                    if (bild is not null && File.Exists(bild.DriveImage))
                    {
                        posBuilder.AppendLine("<tr>");
                        posBuilder.AppendLine("\t<td colspan=\"7\">");
                        if (pos.DruckBildInVollerBreite)
                        {
                            posBuilder.AppendLine($"\t\t<img class=\"bild\" src=\"file:\\\\{bild.DriveImage}\" />");
                        }
                        else
                        {
                            posBuilder.AppendLine($"\t\t<img class=\"bild-fixed\" src=\"file:\\\\{bild.DriveImage}\" />");
                        }

                        if (IsEnglisch)
                        {
                            posBuilder.AppendLine($"\t\t<p>Scope of delivery and color may differ from picture.</p>");
                        }
                        else
                        {
                            posBuilder.AppendLine($"\t\t<p>Lieferumfang und Farbe können von Bild abbweichen.</p>");
                        }


                        posBuilder.AppendLine("\t</td>");
                        posBuilder.AppendLine("</tr>");

                    }

                }



                if ((sucheOption is not null && sucheOption.OPTI_L_BERECHNEN) || sucheOption is null)
                {
                    mwst += pos.GetMwstSumme();
                    rabatt += pos.GetRabatt1();
                    rabatt += pos.GetRabatt2();
                    netto += pos.GetGesamtNetto();
                }
            }


            mwst += _angebot.Zusatzkosten * _versandMehrwertsteuer.MWST_N_PROZENT / 100;

            if (_regeln.ShowZwischensumme)
            {
                posBuilder.AppendLine("<tr class=\"total display-zwischensumme border-top\">");
            }
            else
            {
                posBuilder.AppendLine("<tr class=\"total display-zwischensumme\">");
            }

            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tSubtotal:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tZwischensumme:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{(netto - rabatt):0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-versandkosten\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tShipping costs:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tVersandkosten:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{_angebot.Frachtkosten:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-verpackungskosten\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tPacking costs:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tVerpackungskosten:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{_angebot.Verpackungskosten:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-versicherungskosten\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tInsurance costs:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tVersicherungskosten:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{_angebot.Versicherungskosten:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");



            if (_regeln.ShowZwischensumme)
            {
                posBuilder.AppendLine("<tr class=\"total font-weight-bold\">");
            }
            else
            {
                posBuilder.AppendLine("<tr class=\"total border-top font-weight-bold\">");
            }

            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tTotal EUR without VAT.:");
            }
            else
            {
                posBuilder.AppendLine($"\t\tTotal EUR ohne Mwst.:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{netto + _angebot.Zusatzkosten - rabatt:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-mwst\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            posBuilder.AppendLine($"\t\tzzgl. {_angebot.BELE_N_MWST_SATZ1}% MwSt:");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{mwst:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-endbetrag-brutto font-weight-bold\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\tTotal EUR incl. VAT.");
            }
            else
            {
                posBuilder.AppendLine($"\t\tTotal EUR inkl. Mwst.:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t{netto + _angebot.Zusatzkosten - rabatt + mwst:0.00}");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            posBuilder.AppendLine("<tr class=\"total display-rabatt\">");
            posBuilder.AppendLine("\t<td colspan=\"6\" class=\"text-right\">");
            if (IsEnglisch)
            {
                posBuilder.AppendLine($"\t\t(Discount:");
            }
            else
            {
                posBuilder.AppendLine($"\t\t(eingerechneter Rabatt:");
            }
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("\t<td class=\"text-right\">");
            posBuilder.AppendLine($"\t\t-{rabatt:0.00})");
            posBuilder.AppendLine("\t</td>");
            posBuilder.AppendLine("</tr>");

            if(rabatt is 0)
            {
                _regeln.ShowRabatt = false;
            }

            Lieferanschrift? lieferanschrift = null;
            Rechnungsanschrift? rechnungsanschrift = null;
            Lieferbedingung? lieferbedingung = _lieferbedingungen.Where(x => x.LIBD_N_NR == _angebot.LieferbedingungId).FirstOrDefault();
            Zahlungsbedingung? zahlungsbedingung = _zahlungsbedingungen.Where(x => x.ZABD_N_NR == _angebot.ZahlungsbedingungId).FirstOrDefault();
            if (_angebot.LieferanschriftId > 0)
            {
                lieferanschrift = await Lieferanschrift.GetLieferanschriftAsync(_angebot.Kundennummer, _angebot.LieferanschriftId);
            }

            if (_angebot.RechnungsanschriftId > 0)
            {
                rechnungsanschrift = await Rechnungsanschrift.GetRechnungsanschriftAsync(_angebot.Kundennummer, _angebot.RechnungsanschriftId);
            }



            Land land = await Land.GetLandAsync(_angebot.Land, fbController) ?? throw new ArgumentNullException(nameof(land));





            _template = _template
                .Replace("{{ DISPLAY_CSS }}", GenerateDisplayCss())
                .Replace("{{ IMPRESSUM }}", GlobalConfig.Configuration.FirmenDaten.ImpressumHtmlDruckformulare())
                .Replace("{{ ABS_FIRMA }}", GlobalConfig.Configuration.FirmenDaten.Firmenname)
                .Replace("{{ ABS_STRASSE }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Strasse)
                .Replace("{{ ABS_LAND }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Land)
                .Replace("{{ ABS_PLZ }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Postleitzahl)
                .Replace("{{ ABS_ORT }}", GlobalConfig.Configuration.FirmenDaten.Firmenanschrift.Ort)
                .Replace("{{ ADDRESSZEILE1 }}", _angebot.Name1)
                .Replace("{{ ADDRESSZEILE2 }}", _angebot.Name2)
                .Replace("{{ ADDRESSZEILE3 }}", _angebot.Name3)
                .Replace("{{ ADDRESSZEILE4 }}", partner is not null ? partner.GetName() : _angebot.Ansprechpartner)
                .Replace("{{ EMP_STRASSE }}", _angebot.Strasse)
                .Replace("{{ EMP_LAND }}", land.LAND_A_NAME)
                .Replace("{{ EMP_PLZ }}", _angebot.Postleitzahl)
                .Replace("{{ EMP_ORT }}", _angebot.Ort)
                .Replace("{{ DATUM }}", _angebot.BELE_D_DATE.ToShortDateString())
                .Replace("{{ KUNDENNUMMER }}", _angebot.Kundennummer)
                .Replace("{{ ANFRAGE_DURCH }}", _angebot.AnfrageDurch)
                .Replace("{{ ANFRAGE_VOM }}", _angebot.AnfrageVom.ToShortDateString())
                .Replace("{{ GUELTIG_BIS }}", _angebot.GültigBis.ToShortDateString())
                .Replace("{{ KOPFTEXT }}", _angebot.Kopftext)
                .Replace("{{ FUSSTEXT }}", _angebot.Fußtext)
                .Replace("{{ POSITIONEN }}", posBuilder.ToString())
                .Replace("{{ ANGEBOTSNUMMER }}", _angebot.Belegnummer.ToString())
                .Replace("{{ LIEFERANSCHRIFT }}", lieferanschrift?.ToString() ?? String.Empty)
                .Replace("{{ RECHNUNGSANSCHRIFT }}", rechnungsanschrift?.ToString() ?? String.Empty)
                .Replace("{{ LIEFERZEIT }}", _angebot.Lieferzeit)
                .Replace("{{ LIEFERBEDINGUNGEN }}", IsEnglisch
                ? lieferbedingung?.LIBD_A_TEXT5 ?? String.Empty
                : lieferbedingung?.LIBD_A_TEXT1 ?? String.Empty)
                .Replace("{{ ZAHLUNGSBEDINGUNGEN }}", IsEnglisch
                ? zahlungsbedingung?.ZABD_A_TEXT2 ?? String.Empty
                : zahlungsbedingung?.ZABD_A_TEXT1 ?? String.Empty)
                .Replace("{{ ANFRAGENNUMMER }}", _angebot.Anfragennummer)
                ;

            return await Task.FromResult(_template);
        }
    }
}
