﻿using MbeSoap;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using WK5.Core.Models;
using WK5.Core.Models.Versand;

namespace WK5.Core.Basis.Erweiterungen
{
    public static class VersandErweiterungen
    {

        public static async Task<string> GetVersandEmailAdresseAsync(this Beleg beleg, FbController2 fbController)
        {
            //Prüft ob eine Email zur Lieferanschrift existiert und holt sich ggf. eine
            string versandEmailAdresse = string.Empty;
            Lieferanschrift? lieferanschrift = null;
            if (beleg.LieferanschriftId > 0)
            {
                lieferanschrift = await Lieferanschrift.GetLieferanschriftAsync(beleg.Kundennummer, beleg.LieferanschriftId);
            }

            // Neutrale Belege gehen nicht an die Lieferanschrift
            if (lieferanschrift is null || String.IsNullOrEmpty(lieferanschrift.KULA_A_EMAIL) || beleg.NeutralerVersender)
            {
                if (!String.IsNullOrWhiteSpace(beleg.Ansprechpartner))
                {
                    Ansprechpartner? part = await Ansprechpartner.GetAnsprechpartnerByNameAsync(beleg.Kundennummer, beleg.Ansprechpartner);
                    if (part is not null && !String.IsNullOrEmpty(part.PART_A_EMAIL))
                    {
                        versandEmailAdresse = part.PART_A_EMAIL;
                    }
                }

                if (String.IsNullOrWhiteSpace(versandEmailAdresse))
                {
                    Kunde? kunde = await Kunde.GetKundeAsync(beleg.Kundennummer);
                    if (kunde is not null && !String.IsNullOrEmpty(kunde.KUND_A_EMAIL))
                    {
                        versandEmailAdresse = kunde.KUND_A_EMAIL;
                    }
                }
            }
            else
            {
                versandEmailAdresse = lieferanschrift.KULA_A_EMAIL;
            }

            return versandEmailAdresse;
        }

        public static async Task<string> GetRechnungsEmailAdresseAsync(this Beleg beleg, FbController2 fbController)
        {
            string emailAdresse = string.Empty;
            Rechnungsanschrift? rechnungsanschrift = null;
            if (beleg.RechnungsanschriftId > 0)
            {
                rechnungsanschrift = await Rechnungsanschrift.GetRechnungsanschriftAsync(beleg.Kundennummer, beleg.RechnungsanschriftId);
            }

            if (rechnungsanschrift is not null && !String.IsNullOrWhiteSpace(rechnungsanschrift.KURE_A_EMAIL))
            {
                emailAdresse = rechnungsanschrift.KURE_A_EMAIL;
            }

            if (String.IsNullOrWhiteSpace(emailAdresse) && !string.IsNullOrWhiteSpace(beleg.Ansprechpartner))
            {
                Ansprechpartner? ansprechpartner = await Ansprechpartner.GetAnsprechpartnerByNameAsync(beleg.Kundennummer, beleg.Ansprechpartner);
                if (ansprechpartner is not null && !String.IsNullOrWhiteSpace(ansprechpartner.PART_A_EMAIL))
                {
                    emailAdresse = ansprechpartner.PART_A_EMAIL;
                }
            }

            if (String.IsNullOrWhiteSpace(emailAdresse))
            {
                Kunde? kunde = await Kunde.GetKundeAsync(beleg.Kundennummer);
                if (kunde is not null && !String.IsNullOrWhiteSpace(kunde.KUND_A_EMAIL))
                {
                    emailAdresse = kunde.KUND_A_EMAIL;
                }
            }

            return emailAdresse;
        }

        public static async Task<string> GetBelegDateinameAsync(this Beleg beleg, FbController2 fbController)
        {
            Rechnungsanschrift? anschrift = await Rechnungsanschrift.GetRechnungsanschriftAsync(beleg.Kundennummer, beleg.RechnungsanschriftId);
            if (anschrift is null)
            {
                return $"{beleg.Belegtyp}-{beleg.Belegnummer}.pdf";
            }

            string temp = anschrift.KURE_B_NOTIZ ?? String.Empty;
            if (temp.Contains("File="))
            {
                int start = temp.IndexOf("File=");
                temp = temp[start..];
                if (!temp.Contains(".pdf"))
                {
                    return $"{beleg.Belegtyp}-{beleg.Belegnummer}.pdf";
                }
                else
                {
                    int end = temp.IndexOf(".pdf");
                    temp = temp.Remove(end + 4);
                    return temp[5..];
                }
            }
            else
            {
                return $"{beleg.Belegtyp}-{beleg.Belegnummer}.pdf";
            }
        }
        public static RecipientType ToRecipientType(string ansprechpartner, string firma, string name2, string name3, string strasse, string plz, string ort, string land, string telefonnummer, string email)
        {
            RecipientType recipient = new RecipientType();



            if (String.IsNullOrWhiteSpace(recipient.Name))
            {
                recipient.Name = ansprechpartner;
            }

            recipient.Name = recipient.Name.Length > 35 ? recipient.Name.Substring(0, 34) : recipient.Name;
            recipient.CompanyName = firma.Length > 35 ? firma.Substring(0, 34) : firma;

            #region Adresszeilen Splitten

            ReadOnlySpan<char> charsOfAdresse = strasse.AsSpan();
            ReadOnlySpan<char> charsOfBelegname2 = name2.AsSpan();
            ReadOnlySpan<char> charsOfBelegname3 = name3.AsSpan();

            if (charsOfAdresse.Length > 100)
            {
                recipient.Address = charsOfAdresse[0..99].ToString();

                if (charsOfAdresse.Length > 135)
                {
                    recipient.Address2 = charsOfAdresse[100..134].ToString();
                    if (charsOfAdresse.Length > 170)
                    {
                        recipient.Address3 = charsOfAdresse[135..169].ToString();
                    }
                    else
                    {
                        recipient.Address3 = charsOfAdresse[135..(charsOfAdresse.Length - 1)].ToString();
                    }
                }
                else
                {
                    recipient.Address2 = charsOfAdresse[100..(charsOfAdresse.Length - 1)].ToString();
                }
            }
            else
            {
                recipient.Address = charsOfAdresse.ToString();
            }

            if (String.IsNullOrWhiteSpace(recipient.Address2))
            {
                if (charsOfBelegname2.Length > 35)
                {
                    recipient.Address2 = charsOfBelegname2[0..34].ToString();
                    if (charsOfBelegname3.Length > 70)
                    {
                        recipient.Address3 = charsOfBelegname2[35..69].ToString();
                    }
                    else
                    {
                        recipient.Address3 = charsOfBelegname2[35..(charsOfBelegname2.Length - 1)].ToString();
                    }
                }
                else
                {
                    recipient.Address2 = charsOfBelegname2.ToString();
                }
            }

            if (String.IsNullOrWhiteSpace(recipient.Address3))
            {
                if (charsOfBelegname3.Length > 35)
                {
                    recipient.Address3 = charsOfBelegname3[0..34].ToString();
                }
                else
                {
                    recipient.Address3 = charsOfBelegname3.ToString();
                }
            }
            #endregion


            recipient.Phone = telefonnummer;
            if (String.IsNullOrWhiteSpace(recipient.Phone))
            {
                recipient.Phone = GlobalConfig.Configuration.FirmenDaten.Telefon;
            }

            recipient.Email = email;
            if (String.IsNullOrWhiteSpace(recipient.Email))
            {
                recipient.Email = GlobalConfig.Configuration.FirmenDaten.Email;
            }

            recipient.ZipCode = plz;
            recipient.City = ort;
            recipient.Country = land;

            if (land.Length > 2)
            {
                throw new ArgumentException($"Land darf nicht länger als 2 Zeichen sein.", nameof(land));
            }

            if (recipient.Name == recipient.Address2 || recipient.CompanyName == recipient.Address2)
            {
                recipient.Address2 = null;
            }

            if (recipient.Name == recipient.Address3 || recipient.CompanyName == recipient.Address3)
            {
                recipient.Address3 = null;
            }

            if (recipient.Phone.Length > 15)
            {
                string old_phone = recipient.Phone;
                recipient.Phone = old_phone.Substring(0, 15);

            }

            return recipient;
        }

        public static async Task<RecipientType> ToRecipientTypeAsync(this Beleg beleg, FbController2 fbController)
        {

            Kunde kunde = await Kunde.GetKundeAsync(beleg.Kundennummer) ?? throw new NullReferenceException(nameof(kunde));

            if (beleg.LieferanschriftId > 0)
            {
                Lieferanschrift anschrift = await Lieferanschrift.GetLieferanschriftAsync(beleg.Kundennummer, beleg.LieferanschriftId) ?? throw new NullReferenceException(nameof(anschrift));

                string telefonnummer = anschrift.KULA_A_TEL ?? string.Empty;
                string email = anschrift.KULA_A_EMAIL ?? string.Empty;

                if (String.IsNullOrWhiteSpace(telefonnummer))
                {
                    telefonnummer = kunde.KUND_A_TEL1 ?? string.Empty;
                }

                if (String.IsNullOrWhiteSpace(email))
                {
                    email = kunde.KUND_A_EMAIL ?? string.Empty;
                }

                // 12.05.2022: MK: Wir nehmen nicht mehr den Ansprechpartner des Belegs, sondern den Ansprechpartner der Lieferanschrift. Ley hat das so abgesegnet
                return ToRecipientType(
                    ansprechpartner: anschrift.KULA_A_PARTNER ?? String.Empty,
                    firma: anschrift.KULA_A_NAME1,
                    name2: anschrift.KULA_A_NAME2,
                    name3: anschrift.KULA_A_NAME3 ?? string.Empty,
                    strasse: anschrift.KULA_A_STRASSE,
                    plz: anschrift.KULA_A_PLZ,
                    ort: anschrift.KULA_A_ORT,
                    land: anschrift.KULA_A_LAND,
                    telefonnummer: telefonnummer,
                    email: email
                );
            }

            return ToRecipientType(
                ansprechpartner: beleg.Ansprechpartner,
                firma: beleg.Name1,
                name2: beleg.Name2,
                name3: beleg.Name3,
                strasse: beleg.Strasse,
                plz: beleg.Postleitzahl,
                ort: beleg.Ort,
                land: beleg.Land,
                telefonnummer: kunde.KUND_A_TEL1 ?? string.Empty,
                email: kunde.KUND_A_EMAIL ?? string.Empty
             );
        }
    }
}
