﻿using KarleyLibrary.Erweiterungen;
using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Forms;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Models;

namespace WK5_Blazor.Pages.Verkauf.Wartungen
{
    public partial class WartungsForm : IEditPage, IPipeline<Wartung>
    {
        [Parameter] public int Wartungsnummer { get; set; }
        public Wartung Input { get; set; } = new Wartung();
        public Wartung StartCopy { get; set; } = new Wartung();
        public Pipeline<Wartung> Pipeline { get; set; } = new Pipeline<Wartung>();
        public string ActivePageName => Modus switch
        {
            EditMode.Anlage => "Neue Wartung",
            EditMode.Bearbeiten => $"Wartung {Wartungsnummer}",
            _ => throw new ArgumentOutOfRangeException(nameof(Modus))
        };

        public string FinalBreadcrumbItemName => Modus switch
        {
            EditMode.Anlage => "Neue Wartung",
            EditMode.Bearbeiten => $"{Wartungsnummer}",
            _ => throw new ArgumentOutOfRangeException(nameof(Modus))
        };
        public EditMode Modus { get; set; } = EditMode.Anlage;

#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable

        private EditForm? _editForm;

        private bool _keineWartungenMehrFürDiesesGerät;
        private List<Ansprechpartner> _partner = new List<Ansprechpartner>();
        protected override async Task OnParametersSetAsync()
        {
            if (Wartungsnummer > 0)
            {
                await LadeEditModeAsync();
            }

            StartCopy = Input.DeepCopy();

            await SetupPipelineAsync();
            await SetAlerts();
        }
        public async Task LadeEditModeAsync()
        {
            using FbController2 fbController = new FbController2();
            Wartung? wartung = await wartungsService.GetWartungAsync(Wartungsnummer, fbController);

            if (wartung is not null)
            {
                Input = wartung;

                _partner = await Ansprechpartner.GetAnsprechpartnerAsync(wartung.Kundennummer).ToListAsync();
                _keineWartungenMehrFürDiesesGerät = !await wartungsService.WartungenFürSeriennummerEnabled(Input.Kundennummer, Input.Seriennummer, fbController);
                Modus = EditMode.Bearbeiten;

            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Die Wartung {Wartungsnummer} konnte nicht gefunden werden"
                });

                navigationManager.NavigateTo("/Wartungen");
            }
        }
        public async Task SetupPipelineAsync()
        {
            Pipeline = new Pipeline<Wartung>();
            await Pipeline.HardReset();

            Pipeline.Add(async (input) =>
            {
                await SaveAsync();
                return true;
            });
        }
        private async Task SubmitAsync()
        {
            if (_editForm is null || _editForm.EditContext is null)
            {
                return;
            }

            if (_editForm.EditContext.Validate())
            {
                Pipeline.Reset();
                await Pipeline.RunUntilFailureAsync(Input);
            }
        }

        private async Task SaveAsync()
        {
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            if (Modus is EditMode.Anlage)
            {
                // 24.08.2021 - Wartungen werden ausschließlich durch den Cornjob angelegt.
                throw new NotImplementedException();
            }
            else
            {
                await wartungsService.UpdateAsync(Input, fbController);
            }

            if (Modus is EditMode.Anlage)
            {
                navigationManager.NavigateTo($"/Wartungen/{Input.WartungsNr}");
            }
            else
            {
                await OnParametersSetAsync();
            }
        }

        private async Task ÜbernahmeInAngebotAsync()
        {
            if(_editForm is null || _editForm.EditContext is null)
            {
                return;
            }

            if(_editForm.EditContext.Validate())
            {
                Input.Angenommen = true;
                if (Input.AngenommenDatum == default)
                {
                    Input.AngenommenDatum = DateTime.Now;
                }
                await SaveAsync();
                navigationManager.NavigateTo($"/Angebote/Neu/Wartung/{Input.WartungsNr}");
            }
            else
            {
                await jsRuntime.ShowToast(ToastType.error, "Übernahme fehlgeschlagen, da die Wartung nicht gespeichert werden konnte");
            }
        }

        #region AutoComplete
        private Task AngenommenChangedAsync(bool value)
        {
            Input.Angenommen = value;

            if (Input.Angenommen && Input.AngenommenDatum == default)
            {
                Input.AngenommenDatum = DateTime.Now;
            }

            return Task.CompletedTask;
        }

        private Task StorniertChangedAsync(bool value)
        {
            Input.Storniert = value;

            if (Input.Storniert && Input.StorniertDatum == default)
            {
                Input.StorniertDatum = DateTime.Now;
            }

            return Task.CompletedTask;
        }

        private Task ErledigtChangedAsync(bool value)
        {
            Input.Erledigt = value;

            if (Input.Erledigt && Input.ErledigtDatum == default)
            {
                Input.ErledigtDatum = DateTime.Now;
            }

            return Task.CompletedTask;
        }

        private async Task KeineGeräteWartungMehrAsync(bool value)
        {
            _keineWartungenMehrFürDiesesGerät = value;

            using FbController2 fbController = new FbController2();
            if(_keineWartungenMehrFürDiesesGerät)
            {
                await wartungsService.DeaktiviereWartungenFürSeriennummer(Input.Kundennummer, Input.Seriennummer, fbController);
                await jsRuntime.ShowToast(ToastType.success,"Wartungen für dieses Gerät wurden erfolgreich deaktiviert.");
            }
            else
            {
                await wartungsService.EnableWartungenFürSeriennummer(Input.Kundennummer, Input.Seriennummer, fbController);
                await jsRuntime.ShowToast(ToastType.success, "Wartungen für dieses Gerät wurden erfolgreich aktiviert.");
            }
        }

        private async Task KeineKundenWartungenMehrAsync(bool value)
        {
            Input.KeineKundenWartungen = value;

            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            if(Input.KeineKundenWartungen)
            {
                await wartungsService.UpdateKeineKundenWartungen(true, Input.Kundennummer, fbController);
                await jsRuntime.ShowToast(ToastType.success, "Wartungen für den Kunden wurden erfolgreich deaktiviert.");
            }
            else
            {
                await wartungsService.UpdateKeineKundenWartungen(false, Input.Kundennummer, fbController);
                await jsRuntime.ShowToast(ToastType.success, "Wartungen für den Kunden wurden erfolgreich aktiviert.");
            }
        }
        
        #endregion
        #region NotImplemented
        public string GetNavListClass(bool active) => throw new NotImplementedException();

        public string GetSubNavListClass(bool active) => throw new NotImplementedException();

        public string GetTabClass(bool active) => throw new NotImplementedException();
        #endregion
    }
}
