﻿using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Forms;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Models;

namespace WK5_Blazor.Pages.Verkauf.Rechnungen
{
    public partial class WiederkehrendeRechnungForm
    {
#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable
        [Parameter] public int Rechnungsnummer { get; set; }
        private EditContext editContext;
        private EditMode Modus { get; set; } = EditMode.Anlage;

        private WiederkehrendeRechnung Input { get; set; } = new WiederkehrendeRechnung();
        [BindProperty]
        public bool Sonderkündigung { get; set; }
        private WiederkehrendeRechnung Vergleich { get; set; } = new WiederkehrendeRechnung();


        private bool showKündigungsModal;

        public WiederkehrendeRechnungForm()
        {
            editContext = new EditContext(Input);
        }

        private async Task RechnungsnummerChangedAsync(int rechnungsnummer)
        {
            Input.WIED_N_NR = rechnungsnummer;
            using FbController2 fbController = new FbController2();
            fbController.AddParameter("@BELE_N_NR", rechnungsnummer);
            var row = await fbController.SelectRowAsync(@"SELECT KUND_A_NAME1, BELE_N_NR_AU, BELE_D_DATE FROM BELEGE B
LEFT JOIN KUNDEN K ON (K.KUND_A_NR = B.BELE_A_KUNDENNR)
WHERE BELE_A_TYP = 'RE' AND BELE_N_NR = @BELE_N_NR");

            if (row != null)
            {
                Input.KUND_A_NAME1 = row["KUND_A_NAME1"] as string ?? String.Empty;
                int.TryParse(row["BELE_N_NR_AU"].ToString(), out int auftragsnummer);
                Input.BELE_N_NR_AU = auftragsnummer;
                if (Modus == EditMode.Anlage)
                {
                    DateTime.TryParse(row["BELE_D_DATE"].ToString(), out DateTime belegdatum);
                    Input.WIED_D_START = belegdatum;
                }
                else
                {
                    Input.WIED_D_KUENDIGUNGSDATUM = default;
                }
            }
            else
            {
                Input.KUND_A_NAME1 = String.Empty;
                Input.BELE_N_NR_AU = 0;
                Input.WIED_D_START = default;
            }

            var fd = editContext.Field(nameof(Input.CustomerExists));
            editContext.NotifyFieldChanged(fd);
            fd = editContext.Field(nameof(Input.AuftragForRechnungExists));
            editContext.NotifyFieldChanged(fd);
            fd = editContext.Field(nameof(Input.WIED_D_PREISAENDERUNG_ZUM));
            Input.WIED_D_PREISAENDERUNG_ZUM = default;
            editContext.NotifyFieldChanged(fd);
        }

        protected override async Task OnParametersSetAsync()
        {
            if (Rechnungsnummer > 0)
            {
                using FbController2 fbController = new FbController2();
                var wiederkehrendeRechnung = await wiederkehrendeRechnungenService.GetWiederkehrendeRechnungAsync(Rechnungsnummer, fbController);

                if (wiederkehrendeRechnung is not null)
                {
                    Vergleich = (WiederkehrendeRechnung)wiederkehrendeRechnung.Clone();
                    Modus = EditMode.Bearbeiten;
                    Input = wiederkehrendeRechnung;                   
                    editContext = new EditContext(Input);
                }
            }
        }


        private async Task SaveAsync()
        {
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await fbController.StartTransactionAsync();

            try
            {
                switch (Modus)
                {
                    case EditMode.Anlage:
                        if(await wiederkehrendeRechnungenService.WiederkehrendeRechnungExistsAsync(Input.WIED_N_NR, fbController))
                        {
                            await jsRuntime.ShowToast(ToastType.error, "Die wiederkehrende Rechnung konnte nicht angelegt werden, da bereits eine wiederkehrende Recnung für diese Rechnungsnummer angelegt worden ist.");
                            return;
                        }
                        await wiederkehrendeRechnungenService.CreateWiederkehrendeRechnungAsync(Input, Mitarbeiter.PersonalNummer, fbController);
                        await AddAlertAsync(new AlertBox { AlertType = AlertType.Success, Message = "Wiederkehrende Rechnung wurde erfolgreich angelegt." });
                        break;
                    case EditMode.Bearbeiten:
                        await wiederkehrendeRechnungenService.UpdateWiederkehrendeRechnungAsync(Input, Mitarbeiter.PersonalNummer, fbController);
                        await AddAlertAsync(new AlertBox
                        {
                            AlertType = AlertType.Success,
                            Message = "Wiederkehrende Rechnung wurde erfolgreich gespeichert."
                        });
                        break;
                    default:
                        break;
                }
                await fbController.CommitChangesAsync();
            }
            catch (Exception ex)
            {
                await fbController.RollbackChangesAsync();
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Fehler beim speichern der wiederkehrenden Rechnung. Fehler: {ex}"
                });
            }

            navigationManager.NavigateTo("/Rechnung/WiederkehrendeRechnungen");
        }
        private async Task WiederkehrendeRechnungKündigen()
        {
            if (Input is not null && Modus is EditMode.Bearbeiten)
            {
                using FbController2 fbController = new FbController2();
                if (Sonderkündigung)
                {
                    // Wir nehmen des aktuelle Datum -1, da keine Rechnungen mehr erstellt werden, wenn das Kündigungsdatum < als der aktuelle Tag ist.
                    // Bei einer Sonderkündigung werden keine weiteren Rechnungen mehr geschrieben.
                    await wiederkehrendeRechnungenService.WiederkehrendeRechnungKündigenAsync(Input.WIED_N_NR, DateTime.Now.AddDays(-1), fbController);
                    await Input.AufInaktivAsync(fbController);
                }
                else
                {
                    await wiederkehrendeRechnungenService.WiederkehrendeRechnungKündigenAsync(Input.WIED_N_NR, fbController);
                }


                await AddAlertAsync(new AlertBox { AlertType = AlertType.Success, Message = "Die Wiederkehrende Rechnung wurde erfolgreich gekündigt." });


                navigationManager.NavigateTo("/Rechnung/WiederkehrendeRechnungen");
            }
        }
    }
}
