﻿using KarleyLibrary.Erweiterungen;
using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Forms;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using Toolbelt.Blazor.HotKeys;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Drucken.Lieferscheine;
using WK5.Core.Email;
using WK5.Core.Models;
using WK5.Core.Models.Tools.Lagerregal;
using WK5.Core.PageModels.Verkauf.Lieferscheine;

namespace WK5_Blazor.Pages.Verkauf.Lieferscheine
{
    public partial class LieferscheinForm 
    {
        [Parameter] public int Lieferscheinnummer { get; set; }
        [Parameter] public int RmaNummer { get; set; }
        [Parameter] public int AustauschArt { get; set; }
        public Pipeline<Lieferschein> ÜbernahmePipeline { get; set; } = new Pipeline<Lieferschein>();

        private bool _showStücklisteModal;
        private bool _showÜbernahmeModal;
        private bool _showKundenArtikelhistorieModal;
        private bool _showPosLöschenModal;
        private bool _showSeriennummerModal;
        private Queue<Belegposition> _chargenContinueWith = new Queue<Belegposition>();
        private Belegposition? _seriennummerChargenPosition;
        private List<Seriennummer> _chargenSeriennummern = new List<Seriennummer>();
        private Rma? _rma;
        private Func<bool, Task>? _confirmAction;
        private bool _confirmIsMail = false;
        private bool _showAuftragErledigenModal;
        private bool _isCharging;
        public LieferscheinForm()
        {
            Input = new Lieferschein();
            FilterPakete.Typ = PaketTyp.Lieferschein;
            FilterZeiterfassung.Option = ZeitenFilterOption.Lieferschein;
            StartCopy = new Lieferschein();
            ActiveTab = LieferscheinTabs.Kundendaten;
            EditOnlyTabs.Add(LieferscheinTabs.Dokumentenablage);
            EditOnlyTabs.Add(LieferscheinTabs.Historie);
        }
        public override string ActivePageName => Modus switch
        {
            EditMode.Anlage => "Neuer Lieferschein",
            EditMode.Bearbeiten => $"LS-{Lieferscheinnummer}",
            _ => "Unbekannt"
        };
        public override string FinalBreadcrumbItemName => Modus switch
        {
            EditMode.Anlage => "Neu",
            EditMode.Bearbeiten => $"{Lieferscheinnummer}",
            _ => "Unbekannt"
        };

       

        protected override async Task OnParametersSetAsync()
        {
            _versandMehrwertsteuer = await Mehrwertsteuer.GetVersandMehrwertsteuerAsync();

            if (Lieferscheinnummer > 0)
            {
                await LadeEditModeAsync();
            }

            FilterPakete.Nummer = Lieferscheinnummer.ToString();
            FilterZeiterfassung.Suchbegriff = Lieferscheinnummer.ToString();
            FilterZeiterfassung.Kundennummer = Input.Kundennummer;
            StartCopy = Input.DeepCopy();


            if (Modus is EditMode.Anlage && RmaNummer > 0)
            {
                using FbController2 fbController = new FbController2();
                _rma = await rmaService.GetAsync(RmaNummer, fbController);

                if (_rma is not null)
                {
                    Input.RmaNummerVerknüpfung = RmaNummer;
                    if (AustauschArt is 0)
                    {
                        Input.BELE_L_REPARATUR = true;
                    }
                    Kunde = await Kunde.GetKundeAsync(_rma.Kundennummer);
                    if (Kunde is not null)
                    {
                        await KundeSelectedAsync(Kunde);
                    }

                    await AddArtikelAsync(_rma.Artikelnummer);

                    // Die Position wird durch AddArtikelAsync automatisch selected. Jetzt müssen wir noch die Mengen für die RMA anpassen
                    if (SelectedPosition is not null)
                    {
                        SelectedPosition.Menge = _rma.Menge;
                    }

                }
            }

            if (Kunde is null)
            {
                _showKundenSucheModal = true;
            }


            await SetupÜbernahmePipelineAsync();
            await SetAlerts();
        }
        public async override Task LadeEditModeAsync()
        {
            using FbController2 fbController = new FbController2();
            Lieferschein? lieferschein = await Lieferschein.GetLieferscheinAsync(Lieferscheinnummer, fbController);
            if (lieferschein is not null)
            {
                Kunde = await Kunde.GetKundeAsync(lieferschein.Kundennummer);
                if (Kunde is not null)
                {
                    await KundeSelectedAsync(Kunde);

                    Input = lieferschein;
                    Gesperrt = lieferschein.RechnungsnummerVerknüpfung > 0;

                    _zähler = lieferschein.Positionen.Count;
                    await LieferanschriftChangedAsync(Input.LieferanschriftId);
                    await RechnungsanschriftChangedAsync(Input.RechnungsanschriftId);
                    await PositionSelected(Input.Positionen.FirstOrDefault());
                    await LieferbedingungSelectedAsync(Input.LieferbedingungId);
                    _subBelege = await belegService.GetSubBelege(BelegTyp.Lieferschein, Lieferscheinnummer, fbController);
                    Modus = EditMode.Bearbeiten;
                }
            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Lieferschein {Lieferscheinnummer} konnte nicht gefunden werden."
                });


                navigationManager.NavigateTo("/Lieferscheine");
            }

            if (!_isDisposed)
            {
                await CheckActivePageAsync();
            }
        }
        public async Task SetupÜbernahmePipelineAsync()
        {
            await ÜbernahmePipeline.HardReset();

            ÜbernahmePipeline.Add((input) =>
            {
                if(Input.AuftragsnummerVerknüpfung is 0)
                {
                    return true;
                }

                if(Input.BenötigtChargen)
                {
                    _showAuftragErledigenModal = true;
                    return false;
                }
                
                return true;
            });

            ÜbernahmePipeline.Add(async (input) =>
            {
                await ÜbernahmeAsync();
                return true;
            });
        }
        private async Task SubmitAsync()
        {
            if((NurDurchAdminZuBearbeiten && !Mitarbeiter.IsAdmin) || SperreDurchAnderenUser || !Input.HasBeenModified(StartCopy))
            {
                return;
            }


            if (_editForm is null || _editForm.EditContext is null || Kunde is null)
            {
                return;
            }

            if (_editForm.EditContext.Validate())
            {
                using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
                await fbController.StartTransactionAsync();
                try
                {
                    if (Modus is EditMode.Anlage)
                    {
                        await belegService.CreateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);
                        await AddAlertAsync(new AlertBox
                        {
                            AlertType = AlertType.Success,
                            Message = $"Lieferschein {Input.Belegnummer} erfolgreich angelegt",
                            DecayTime = 10
                        });
                    }
                    else
                    {
                        if (Gesperrt && !Mitarbeiter.IsAdmin)
                        {

                            await belegService.UpdateGesperrtenBelegAsync(Input, fbController);
                            await AddAlertAsync(new AlertBox
                            {
                                AlertType = AlertType.Success,
                                Message = $"Notiz und Einkaufspreise wurde erfolgreich gespeichert",
                                DecayTime = 10
                            });
                        }
                        else
                        {
                            await belegService.UpdateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);

                            EmailController emailController = new EmailController();
                            foreach (var pos in Input.Positionen.Where(x => x.ARTI_L_MIETE))
                            {
                                if(!await artikelService.IstTerminVerfügbarAsync(pos, fbController))
                                {
                                    await emailController.SendenAsync(GlobalConfig.EmailInfoEU, "Überschneidung bei Mieten", $@"Für den Artikel <a href=""/Artikel/{pos.Artikelnummer}"">{pos.Artikelnummer}</a> wurde eine Überschneidung der Mietzeiträume festgestellt.<br /><br />
Bitte prüfe die Termine im Artikel und spreche ggf. mit dem Kunden einen Ausweichtermin ab.");
                                }
                            }

                            await AddAlertAsync(new AlertBox
                            {
                                AlertType = AlertType.Success,
                                Message = $"Lieferschein {Input.Belegnummer} erfolgreich gespeichert",
                                DecayTime = 10
                            });
                        }
                    }

                    // Reparatur Belege dürfen nicht die Austauschart ersetzen
                    if (_rma is not null && AustauschArt > 0)
                    {
                        await rmaService.SetAustauschTypAsync(_rma.Id, AustauschArt, fbController);
                    }
                    await fbController.CommitChangesAsync();
                }
                catch (Exception ex)
                {
                    Debug.WriteLine(ex);
                    await fbController.RollbackChangesAsync();
                    throw;
                }

                if (Modus is EditMode.Anlage)
                {
                    navigationManager.NavigateTo($"/Lieferscheine/{Input.Belegnummer}");
                }
                else
                {
                    await OnParametersSetAsync();
                }


            }


        }
        private async Task ÜbernahmeAsync()
        {

            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            //await fbController.StartTransactionAsync();
            try
            {
                Lieferschein? lieferschein = await Lieferschein.GetLieferscheinAsync(Lieferscheinnummer, fbController);

                if (lieferschein is not null && Kunde is not null)
                {
                    await CheckÜbernahmeKundeAsync(lieferschein, Kunde);
                    Rechnung neueRechung = (Rechnung)await belegService.ÜbernahmeAsync(lieferschein, Kunde, Program.AppDaten.Optionen, Program.AppDaten.VersandMehrwertsteuer, fbController, Mitarbeiter.PersonalNummer);
                    await belegService.SetBerechnetAsync(lieferschein, fbController);

                    await AddAlertAsync(new AlertBox
                    {
                        AlertType = AlertType.Success,
                        Message = $"LS-{Lieferscheinnummer} wurde erfolgreich in RE-{neueRechung.Belegnummer} übernommen",
                        DecayTime = 10
                    });

                    //await fbController.CommitChangesAsync();
                    navigationManager.NavigateTo($"/Rechnungen/{neueRechung.Belegnummer}");
                }
            }
            catch (Exception)
            {
                await fbController.RollbackChangesAsync();
                throw;
            }



        }

        private async Task KopierenAsync()
        {
            if (Modus is EditMode.Bearbeiten)
            {
                using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
                await fbController.StartTransactionAsync();
                try
                {

                    Lieferschein? lieferschein = await Lieferschein.GetLieferscheinAsync(Lieferscheinnummer, fbController);
                    if (lieferschein is null)
                    {
                        throw new ArgumentNullException(nameof(lieferschein));
                    }

                    if (Kunde is null)
                    {
                        throw new ArgumentNullException(nameof(Kunde));
                    }

                    var (NeuerBeleg, Messages) = await belegService.CopyAsync(lieferschein, Kunde, Program.AppDaten.Optionen, fbController, Mitarbeiter.PersonalNummer);
                    await AddAlertsAsync(Messages);
                    if (NeuerBeleg is not null)
                    {
                        await fbController.CommitChangesAsync();
                        await AddAlertAsync(new AlertBox
                        {
                            AlertType = AlertType.Success,
                            Message = "Lieferschein wurde erfolgreich kopiert.",
                            DecayTime = 10
                        });

                        navigationManager.NavigateTo($"/Lieferscheine/{NeuerBeleg.Belegnummer}");
                    }
                }
                catch (Exception)
                {
                    await fbController.RollbackChangesAsync();
                    throw;
                }


            }
        }

        #region Chargen
        private async Task<bool> IstSeriennummerChargenVerfügbarAsync()
        {
            if (_editForm is null || _editForm.EditContext is null || Kunde is null)
            {
                return false;
            }

            if (!_editForm.EditContext.Validate())
            {
                await jsRuntime.ShowToast(ToastType.error, "Chargen ist zurzeit nicht möglich, da der Lieferschein aufgrund eines Eingabefehlers nicht gespeichert werden konnte.");
                return false;
            }

            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await fbController.StartTransactionAsync();
            await belegService.UpdateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);
            await fbController.CommitChangesAsync();

            return true;
        }
        private async Task AutoChargenAsync(Belegposition pos)
        {
            if (_editForm is null || _editForm.EditContext is null || Kunde is null)
            {
                return;
            }
            if (!pos.PositionIstStückliste && !pos.IstBundle && pos.PositionBenötigtSeriennummer)
            {
                return;
            }

            if (!_editForm.EditContext.Validate())
            {
                await jsRuntime.ShowToast(ToastType.error, "Chargen ist zurzeit nicht möglich, da der Lieferschein aufgrund eines Eingabefehlers noch nicht gespeichert werden kann.");
                return;
            }

            
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await fbController.StartTransactionAsync();
            await belegService.UpdateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);
            await fbController.CommitChangesAsync();
            await fbController.StartTransactionAsync();
            try
            {
                pos.PosEinkaufspreis = await chargenService.AutoChargenAsync(pos, Input.RmaNummerVerknüpfung > 0, fbController);
                await fbController.CommitChangesAsync();
                await jsRuntime.ShowToast(ToastType.success, "Position erfolgreich gechargt");
            }
            catch (ChargenException ex)
            {
                await jsRuntime.ShowToast(ToastType.error, $"{ex.Artikelnummer} konnte nicht ausreichend gechargt werden, da nur {ex.VerfügbareMenge} Chargen zur Verfügung stehen.");
                await fbController.RollbackChangesAsync();
            }
            catch (Exception)
            {
                await fbController.RollbackChangesAsync();
                throw;
            }
        }
        private async Task SeriennummerChargenAsync(Seriennummer seriennummer, Belegposition pos)
        {
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await chargenService.SeriennummerChargenAsync(seriennummer, pos, Mitarbeiter.PersonalNummer, Input.Name1, fbController);
            _chargenSeriennummern.Remove(seriennummer);
            //pos.Seriennummern.Add(seriennummer.Nummer);

            if (pos.Gechargt == pos.Menge)
            {
                if (_chargenContinueWith.Any())
                {
                    await LadeSeriennummernAsync();
                }
                else
                {
                    _chargenSeriennummern.Clear();
                }
            }
        }
        private async Task RemoveChargenAsync(Belegposition pos)
        {
            if (Kunde is null)
            {
                return;
            }
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await fbController.StartTransactionAsync();
            await belegService.UpdateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);
            try
            {
                await chargenService.RemoveBelegChargenAsync(pos, Mitarbeiter.PersonalNummer, fbController);
                await fbController.CommitChangesAsync();
                pos.Seriennummern.Clear();
                await jsRuntime.ShowToast(ToastType.success, "Chargen erfolgreich gelöscht");
            }
            catch (Exception)
            {
                await fbController.RollbackChangesAsync();
                throw;
            }
        }
        private Task SetupChargenPositionenAsync(Belegposition pos)
        {
            if (pos.PositionIstStückliste || pos.IstBundle)
            {
                foreach (var subPos in pos.GetEndPositionen().Where(x => x.PositionBenötigtSeriennummer && x.Gechargt < x.Menge))
                {
                    _chargenContinueWith.Enqueue(subPos);
                }
            }
            else
            {
                _chargenContinueWith.Enqueue(pos);
            }

            return Task.CompletedTask;
        }
        private async Task LadeSeriennummernAsync()
        {
            if (!_chargenContinueWith.Any())
            {
                await jsRuntime.ShowToast(ToastType.error, "Für die Position wurden bereits alle Seriennummern gechargt.");
                return;
            }
            _seriennummerChargenPosition = _chargenContinueWith.Dequeue();
            if (_seriennummerChargenPosition is not null)
            {
                _chargenSeriennummern = await Seriennummer.FindeOffeneSeriennummernByArtikelAsync(_seriennummerChargenPosition.Artikelnummer).ToListAsync();

                if (_chargenSeriennummern.Count is 0)
                {
                    await jsRuntime.ShowToast(ToastType.error, $"Für den Artikel {_seriennummerChargenPosition.Artikelnummer} konnten keine offenen Seriennummern gefunden werden.");
                }
            }
        }
        #endregion
        #region Download & Mail
        public async override Task DownloadAsync(bool confirmed = false)
        {
            if (Kunde is null)
            {
                return;
            }

            if (!confirmed && Input.BenötigtChargen)
            {
                _confirmAction = DownloadAsync;
                _confirmIsMail = false;
                return;
            }

            _isDownloading = true;
            StateHasChanged(); // Wird für den Shortcut benötigt
            using FbController2 fbController = new FbController2();
            var lieferschein = await Lieferschein.GetLieferscheinAsync(Lieferscheinnummer, fbController);

            if (lieferschein is not null)
            {

                PrintLieferscheinRegelsatz printRegeln = new PrintLieferscheinRegelsatz
                {
                    ShowFooter = true,
                    ShowHeader = true
                };

                if (Program.AppDaten.VersandMehrwertsteuer is null)
                {
                    throw new ArgumentNullException(nameof(Program.AppDaten.VersandMehrwertsteuer));
                }

                PrintLieferschein printer = await PrintLieferschein.CreateAsync(lieferschein, printRegeln);
                string filenamePdf = printer.Print(GlobalConfig.Configuration.OutputPfad);
                await belegService.SetDruckdatumAsync(BelegTyp.Lieferschein, Lieferscheinnummer, fbController);

                await downloadService.DownloadFile($"LS-{Lieferscheinnummer}.pdf", await File.ReadAllBytesAsync(filenamePdf), "application/pdf");
                await downloadService.ClearBuffers();
            }

            _isDownloading = false;
        }
        public async override Task ÖffneMailAsync(bool confirmed = false)
        {

            if (Kunde is null)
            {
                return;
            }

            if (!confirmed && Input.BenötigtChargen)
            {
                _confirmAction = ÖffneMailAsync;
                _confirmIsMail = true;
                return;
            }

            if (Kunde.KUND_A_EMAIL is null)
            {
                await jsRuntime.ShowToast(ToastType.error, "Lieferschein konnte nicht in David geöffnet werden, da beim Kunden keine Mailadresse hinterlegt ist");
                return;
            }
            _mailIsLoading = true;
            StateHasChanged(); // Wird für den Shortcut benötigt
            using FbController2 fbController = new FbController2();
            var lieferschein = await Lieferschein.GetLieferscheinAsync(Lieferscheinnummer, fbController);

            if (lieferschein is not null)
            {

                string empfängerEmail = Kunde.KUND_A_EMAIL;
                if (lieferschein.LieferanschriftId > 0)
                {
                    Lieferanschrift? lieferanschrift = await Lieferanschrift.GetLieferanschriftAsync(lieferschein.Kundennummer, lieferschein.LieferanschriftId);

                    if (lieferanschrift is not null && !String.IsNullOrWhiteSpace(lieferanschrift.KULA_A_EMAIL))
                    {
                        empfängerEmail = lieferanschrift.KULA_A_EMAIL;
                    }
                }


                KarleyBrowserInterfaceEmail email = new KarleyBrowserInterfaceEmail
                {
                    Absender = GlobalConfig.EmailInfoEU,
                    Empfänger = empfängerEmail,
                    Content = String.Empty,
                    Betreff = $"Lieferschein {Lieferscheinnummer}"
                };

                PrintLieferscheinRegelsatz printRegeln = new PrintLieferscheinRegelsatz
                {
                    ShowFooter = true,
                    ShowHeader = true
                };

                if (Program.AppDaten.VersandMehrwertsteuer is null)
                {
                    throw new ArgumentNullException(nameof(Program.AppDaten.VersandMehrwertsteuer));
                }

                PrintLieferschein printer = await PrintLieferschein.CreateAsync(lieferschein, printRegeln);
                string filenamePdf = printer.Print(GlobalConfig.Configuration.OutputPfad);
                await belegService.SetDruckdatumAsync(BelegTyp.Lieferschein, Lieferscheinnummer, fbController);
                email.Anhänge.Add(filenamePdf);

                var (success, data, _) = KarleyLibrary.Serialization.XMLWriter.Serialize(email);
                if (success)
                {
                    string xmlFilename = $"{email.GetHashCode()}.xml";
                    await File.WriteAllTextAsync(Path.Combine(GlobalConfig.Configuration.OutputPfad, xmlFilename), data);
                    await jsRuntime.OpenNewTab($"karley:davidMail;{xmlFilename}");
                }


            }
            _mailIsLoading = false;

        }
        #endregion
        #region Positionen         
        public override string GetPositionNavListClass(Belegposition pos)
        {
            if (!_posValidator.Validate(pos).IsValid)
            {
                return $"{GetSubNavListClass(SelectedPosition == pos)} bg-danger text-light";
            }
            if (pos.BenötigtWeitereSeriennummern || pos.BenötigtWeitereChargen)
            {
                return $"{GetSubNavListClass(SelectedPosition == pos)} bg-secondary text-light";
            }

            return GetSubNavListClass(SelectedPosition == pos);
        }
        private bool IstPositionsMengeGesperrt(Belegposition pos)
        {
            if (pos.BPOS_N_POSID_AU > 0 && pos.Gechargt > 0)
            {
                if (pos.ARTI_L_LAGERFUEHR)
                {
                    return true;
                }
                else if (pos.PositionIstStückliste)
                {
                    return true;
                }
            }

            if (pos.Gechargt > 0 && pos.ARTI_L_LAGERFUEHR)
            {
                return true;
            }

            return false;
        }
        private bool IstPositionGesperrt(Belegposition pos)
        {
            if (Gesperrt)
            {
                return true;
            }

            if (pos.BPOS_N_POSID_AU > 0)
            {
                return true;
            }

            if (pos.ARTI_L_LAGERFUEHR && pos.Gechargt > 0)
            {
                return true;
            }

            return false;
        }

        #endregion

        public override async Task CheckActivePageAsync()
        {
            var (success, page) = ActivePages.AddActivePage(new ActivePage(PageType.Lieferschein, Lieferscheinnummer.ToString(), Mitarbeiter.PersonalNummer));

            if (success)
            {
                Page = page;
            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Der Lieferschein wird zurzeit durch {Program.AppDaten.GetMitarbeiterName(page.UserId)} bearbeitet."
                });

                SperreDurchAnderenUser = true;
            }
        }
        private async Task UploadLieferscheinAsync(InputFileChangeEventArgs e)
        {
            string? extension = Path.GetExtension(e.File.Name);

            if (extension is not null && extension.Equals(".PDF", StringComparison.OrdinalIgnoreCase))
            {
                if (e.File.Size > 5000000)
                {
                    await jsRuntime.ShowToast(ToastType.error, $"Die Datei darf nicht größer als 5MB sein");
                    return;
                }
                byte[] buffer = new byte[e.File.Size];
                await using MemoryStream stream = new MemoryStream(buffer);
                await e.File.OpenReadStream(e.File.Size).CopyToAsync(stream);
                Input.NeutralerLieferschein = buffer;
            }
        }

        private async Task DownloadNeutralerLieferscheinAsync()
        {
            if (Input.NeutralerLieferschein.Length > 0)
            {
                await downloadService.DownloadFile($"NLS_{Lieferscheinnummer}.pdf", Input.NeutralerLieferschein, "application/pdf");
                await downloadService.ClearBuffers();
            }
        }

        public async Task OnTabChange(LieferscheinTabs newActiveTab)
        {
            switch (newActiveTab)
            {
                case LieferscheinTabs.Positionen:
                    if (Input.Positionen.Count is 0)
                    {
                        _showArtikelSucheModal = true;
                    }
                    break;
                case LieferscheinTabs.Historie:
                    await LadeHistorieAsync();
                    break;
                default:
                    break;
            }
        }

        public override void AddHotKeys(HotKeysContext context)
        {
            context.Add(ModKeys.Alt, Keys.S, SubmitAsync, "Speichert den Auftrag", Exclude.Default);
            base.AddHotKeys(context);
        }

        private async Task ConfirmAuftragAufErledigtAsync()
        {
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await belegService.AuftragAufErledigtSetzenAsync(Input.AuftragsnummerVerknüpfung, Mitarbeiter.PersonalNummer, fbController);
            await ÜbernahmePipeline.RunUntilFailureAsync(Input);
        }

        private async Task DeclineAuftragAufErledigtAsync()
        {
            await ÜbernahmePipeline.RunUntilFailureAsync(Input);
        }
    }
}


