﻿using BlazorDownloadFile;
using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Forms;
using Microsoft.JSInterop;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Toolbelt.Blazor.HotKeys;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;
using WK5.Core.Models.VatChecker;
using WK5.Core.PageModels.Stammdaten.Kunden;
using WK5.Core.PageModels.Tools.Gewichtsberechner;
using WK5.Core.PageModels.Tools.Versand;
using WK5.Core.PageModels.Verkauf.Angebote;
using WK5.Core.PageModels.Verkauf.Aufträge;
using WK5.Core.PageModels.Verkauf.Gutschriften;
using WK5.Core.PageModels.Verkauf.Lieferscheine;
using WK5.Core.PageModels.Verkauf.Rechnungen;
using WK5.Core.Services;
using WK5.Core.Tools.Gewichtsberechner;
using WK5.Core.Validators.Verkauf;
using WK5_Blazor.CoreExtensions;

namespace WK5_Blazor.Pages.Verkauf
{
    public abstract class BelegFormBase<TBeleg, TEnumTabs> : ActivePageBase, IEditPage where TBeleg : Beleg where TEnumTabs : System.Enum
    {
#nullable disable
        [Inject] protected NavigationManager navigationManager { get; set; }
        [Inject] protected IJSRuntime jsRuntime { get; set; }
        [Inject] protected IBlazorDownloadFileService downloadService { get; set; }
        [Inject] protected VersandService versandService { get; set; }
        [Inject] protected BelegService belegService { get; set; }
        [Inject] protected VatService vatService { get; set; }
        [Inject] protected ArtikelService artikelService { get; set; }
        [Inject] protected BestellService bestellService { get; set; }
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
        public TBeleg Input { get; set; }
        public TBeleg StartCopy { get; set; }
        public TEnumTabs ActiveTab { get; set; }
        public List<TEnumTabs> EditOnlyTabs { get; set; } = new List<TEnumTabs>();
        public List<TEnumTabs> AdminOnlyTabs { get; set; } = new List<TEnumTabs>();
#nullable enable        
        public Kunde? Kunde { get; set; }
        public BelegTyp BelegTyp => Input switch
        {
            Angebot => BelegTyp.Angebot,
            Auftrag => BelegTyp.Auftrag,
            Lieferschein => BelegTyp.Lieferschein,
            Rechnung => BelegTyp.Rechnung,
            Gutschrift => BelegTyp.Gutschrift,
            _ => throw new InvalidOperationException()
        };
        public Belegposition? SelectedPosition { get; set; }
        public Lieferanschrift? Lieferanschrift { get; set; }
        public Rechnungsanschrift? Rechnungsanschrift { get; set; }
        public KundenÜbersichtFilter KundenFilter { get; set; } = new KundenÜbersichtFilter() { GesperrteKundenSuchen = false };
        public ArtikelFilter ArtikelFilter { get; set; } = new ArtikelFilter();
        public bool Gesperrt { get; set; }
        public bool GesperrtDurchZahlung { get; set; }
        public bool NurDurchAdminZuBearbeiten { get; set; }
        public List<ArtikelZubehör> Zubehör { get; set; } = new List<ArtikelZubehör>();
        public List<Artikel> HistorischeKundenArtikel { get; set; } = new List<Artikel>();
        protected List<AlertBox> VersandHinweise { get; set; } = new List<AlertBox>();

        protected PaketFilter FilterPakete { get; set; } = new PaketFilter();
        protected ZeitenFilter FilterZeiterfassung { get; set; } = new ZeitenFilter();

        public string? ArtikelHinweisText { get; set; } = null;

        protected bool _showKundenSucheModal;
        protected bool _showKundenInfoModal;
        protected bool _lädtVersandkosten;
        protected bool _showArtikelSucheModal;
        protected bool _showLieferzeitTextModal;
        protected bool _isDownloading;
        protected bool _mailIsLoading;
        protected bool _showBelegKopierenModal;
        protected bool _showVersandNeuBerechnenModal;
        protected bool _showTextvorlagenModal;
        protected bool _showPaketeModal;
        protected bool _showZeiterfassungModal;
        protected bool _showTermineModal;
        protected bool _showBundleModal;
        protected int _zähler = 0;
        protected bool _lädtHistorie;
        protected Dictionary<BelegTyp, List<int>> _subBelege = new Dictionary<BelegTyp, List<int>>();
        protected Dictionary<string, decimal> _shippingOptions = new Dictionary<string, decimal>();
        protected BelegpositionValidator _posValidator = new BelegpositionValidator();
        protected List<BelegChangeAnsicht> _historie = new List<BelegChangeAnsicht>();
        protected Belegposition? _letzterKundenpreisPostition;
        protected Mehrwertsteuer? _versandMehrwertsteuer;
        protected Artikel? _artikelwerte;
        protected EditForm? _editForm;
        protected string? _selectedKundennummer;
        protected bool _showKundenkontakte;

        protected List<Ansprechpartner> _ansprechpartner = new List<Ansprechpartner>();
        protected List<Lieferanschrift> _lieferanschriften = new List<Lieferanschrift>();
        protected List<Rechnungsanschrift> _rechnungsanschriften = new List<Rechnungsanschrift>();
        protected List<int> _posInOffenenAufträgen = new List<int>();

        protected List<int> SelectedPositionBedarfsAufträge { get; set; } = new List<int>();
        protected List<int> SelectedPositionBestellungen { get; set; } = new List<int>();
        protected Lieferbedingung? SelectedLieferbedingung { get; set; }

        #region Positionen
        public virtual string GetPositionNavListClass(Belegposition pos)
        {
            if (!_posValidator.Validate(pos).IsValid)
            {
                return $"{GetSubNavListClass(SelectedPosition == pos)} bg-danger text-light";
            }

            return GetSubNavListClass(SelectedPosition == pos);
        }
        protected bool CanMoveUp(Belegposition pos)
        {
            int index = Input.Positionen.IndexOf(pos);
            return index > 0;
        }
        protected bool CanMoveDown(Belegposition pos)
        {
            return Input.Positionen.IndexOf(pos) < _zähler - 1 - pos.BundleArtikel.Count;
        }
        protected Task MovePosition(Belegposition pos, bool moveUp)
        {
            int posIndex = Input.Positionen.IndexOf(pos);
            int vorherigerIndex = posIndex;
            var positionenArray = Input.Positionen.ToArray();
            int letzterBundleIndex = pos.IstBundle ? Array.IndexOf(positionenArray, pos.BundleArtikel.Last()) : vorherigerIndex;

            var array1 = positionenArray[0..vorherigerIndex].ToList();
            var array2 = positionenArray[vorherigerIndex..(letzterBundleIndex + 1)].ToList();
            var array3 = positionenArray[(letzterBundleIndex + 1)..].ToList();

            if (moveUp)
            {
                bool allMoved = false;
                int iterator = array1.Count - 1;
                while (!allMoved)
                {
                    if (array1[iterator].BundleId == 0)
                    {
                        array3.Insert(0, array1[iterator]);
                        array1.RemoveAt(iterator);
                        allMoved = true;
                    }
                    else
                    {
                        array3.Insert(0, array1[iterator]);
                        array1.RemoveAt(iterator);
                        // Wenn array1 leer ist, dann haben wir bereits alles verschoben
                        // Ansonsten gehen wir solange durch,bis wir auf keine weitere BundleId mehr treffen
                        if (iterator - 1 == array1.Count || array1.Count is 0 || array1[iterator - 1].BundleId == 0)
                        {
                            allMoved = true;
                        }
                    }

                    iterator--;
                }



                array1.AddRange(array2);
                array1.AddRange(array3);
            }
            else
            {
                bool allMoved = false;
                int iterator = 0;
                while (!allMoved)
                {
                    if (array3[iterator].BundleId == 0)
                    {
                        array1.Add(array3[iterator]);
                        allMoved = true;
                    }
                    else
                    {
                        array1.Add(array3[iterator]);

                        if (iterator + 1 == array3.Count || array3[iterator + 1].BundleId == 0)
                        {
                            allMoved = true;
                        }
                    }

                    iterator++;
                }

                array3.RemoveRange(0, iterator);

                array1.AddRange(array2);
                array1.AddRange(array3);
            }


            // Fix pos numbers.
            for (int i = 0; i < array1.Count; i++)
            {
                array1[i].PosNr = i + 1;
            }

            Input.Positionen = array1;

            return Task.CompletedTask;
        }
        protected async Task PositionSelected(Belegposition? pos)
        {
            Zubehör.Clear();
            if (Kunde is null)
            {
                return;
            }

            SelectedPositionBedarfsAufträge.Clear();
            SelectedPositionBestellungen.Clear();

            SelectedPosition = pos;
            StateHasChanged();

            if (SelectedPosition is not null && !SelectedPosition.Artikelnummer.Equals("TEXT", StringComparison.OrdinalIgnoreCase))
            {
                using FbController2 fbController = new FbController2();
                _artikelwerte = await artikelService.GetAsync(SelectedPosition.Artikelnummer, fbController);
                _letzterKundenpreisPostition = await belegService.GetLetztenKundenpreisAsync(SelectedPosition.Artikelnummer, Kunde.KUND_A_NR);
                await LadeArtikelzubehör();

                SelectedPositionBedarfsAufträge = await artikelService.GetBelegBedarf(SelectedPosition.Artikelnummer).ToListAsync();
                SelectedPositionBestellungen = await bestellService.GetOffeneBestellungenFürArtikelAsync(SelectedPosition.Artikelnummer).ToListAsync();
                if (SelectedPosition.BPOS_A_TYP is "AU" && SelectedPosition.Artikelnummer != "TEXT")
                {
                    _posInOffenenAufträgen = await belegService.GetOffeneAuftragsnummernFürPosition(SelectedPosition, Kunde.KUND_A_NR, fbController);
                }


            }
            else
            {
                _artikelwerte = null;
                _letzterKundenpreisPostition = null;
                _posInOffenenAufträgen = new List<int>();
            }
        }
        protected Task AddTextAsync()
        {
            Belegposition newPos = new Belegposition
            {
                PosNr = ++_zähler,
                Artikelnummer = "TEXT",
                BPOS_A_TYP = EnumHelper.GetBelegTypString(BelegTyp)
            };

            Input.Positionen.Add(newPos);

            return PositionSelected(newPos);
        }
        protected async Task<Belegposition?> AddArtikelAsync(Artikel artikel)
        {
            if (Kunde is null)
            {
                return null;
            }

            _showArtikelSucheModal = false;

            string landIso2 = Input.Land;
            if (Lieferanschrift is not null)
            {
                landIso2 = Lieferanschrift.KULA_A_LAND;
            }

            Land? land = Program.AppDaten.Länder.Where(x => x.LAND_A_ID == landIso2).FirstOrDefault();

            // 04.07.2022: Die Regelung greift für jedes Land außerhalb von DE
            // 04.07.2022: Bundles berücksichtigt
            if (land is not null && land.LAND_A_ID != "DE" && artikel.ZolltarifId is 0 && (artikel.ARTI_L_LAGERFUEHR || artikel.IstBundle))
            {
                await jsRuntime.ShowToast(ToastType.error, $"Der Artikel {artikel.Artikelnummer} konnte nicht hinzugefügt werden, da dass Land des Empfängers eine Zolltarifnummer erfordert. Hinterlegen Sie im Artikel eine Zolltarifnummer und versuchen Sie es erneut.");
                await jsRuntime.OpenNewTab($"/Artikel/{artikel.Artikelnummer}");
                return null;
            }

            if (land is not null && land.LAND_A_ID != "DE" && string.IsNullOrWhiteSpace(artikel.ARTI_A_URSPRUNGSLAND) && (artikel.ARTI_L_LAGERFUEHR || artikel.IstBundle))
            {
                await jsRuntime.ShowToast(ToastType.error, $"Der Artikel {artikel.Artikelnummer} konnte nicht hinzugefügt werden, da dass Land des Empfängers die Angabe des Ursprunglands des Artikels erfordert. Hinterlegen Sie im Artikel das Ursprungsland und versuchen Sie es erneut.");
                await jsRuntime.OpenNewTab($"/Artikel/{artikel.Artikelnummer}");
                return null;
            }


            Kundenrabatt? kundenpreise = await Kundenrabatt.GetKundenSachnummerAsync(Kunde.KUND_A_NR, artikel.Artikelnummer);






            ArtikelBild? hauptbild = await Artikel.GetArtikelHauptImageAsync(artikel.Artikelnummer);

            var pos = new Belegposition
            {
                PosNr = ++_zähler,
                PosEinkaufspreis = artikel.DurchschnittsEinkaufspreis > 0 ? artikel.DurchschnittsEinkaufspreis : artikel.ARTI_N_EK,
                Gewicht = artikel.ARTI_N_GEWICHT,
                Artikelnummer = artikel.Artikelnummer,
                Bezeichnung1 = artikel.Bezeichnung1,
                Bezeichnung2 = artikel.Bezeichnung2 ?? String.Empty,
                Bezeichnung3 = artikel.Bezeichnung3 ?? String.Empty,
                Bezeichnung4 = artikel.Bezeichnung4 ?? String.Empty,
                Bezeichnung5 = artikel.Bezeichnung5 ?? String.Empty,
                MwstSatz = artikel.MWST_N_PROZENT,
                Gebinde = artikel.WK5_ARTI_N_GEBINDE,
                AnfrageArtikel = artikel.WK5_ARTI_L_ANFRAGEARTIKEL,
                Vpe = artikel.ARTI_N_COLLI,
                Kundensachnummer = kundenpreise?.KUSA_A_ARTNUMMKUND ?? String.Empty,
                Langtext = artikel.Langtext ?? String.Empty,
                Bild = hauptbild?.WebImage ?? String.Empty,
                BildId = hauptbild?.ARBI_N_NR ?? 0,
                ARTI_L_LAGERFUEHR = artikel.ARTI_L_LAGERFUEHR,
                PositionBenötigtSeriennummer = artikel.ARTI_L_SN,
                PositionIstStückliste = false,//artikel.ArtikelIstStückliste,
                IstAbverkaufsartikel = artikel.ARTI_L_ABVERKAUF,
                BPOS_A_TYP = EnumHelper.GetBelegTypString(BelegTyp),
                BPOS_N_NR = Input.Belegnummer,
                ARTI_L_MIETE = artikel.ARTI_L_MIETE,
                Durchschnittseinkaufspreis = artikel.DurchschnittsEinkaufspreis
            };

            if (artikel.IstBundle)
            {
                pos.BundleId = -1;
                Input.Positionen.Add(pos);
                foreach (var bundleArtikel in artikel.BundleArtikel)
                {
                    var subPos = new Belegposition
                    {
                        PosNr = ++_zähler,
                        PosEinkaufspreis = 0,
                        Gewicht = bundleArtikel.ARTI_N_GEWICHT,
                        Artikelnummer = bundleArtikel.Artikelnummer,
                        Bezeichnung1 = bundleArtikel.Bezeichnung1,
                        Bezeichnung2 = bundleArtikel.Bezeichnung2 ?? String.Empty,
                        Bezeichnung3 = bundleArtikel.Bezeichnung3 ?? String.Empty,
                        Bezeichnung4 = bundleArtikel.Bezeichnung4 ?? String.Empty,
                        Bezeichnung5 = bundleArtikel.Bezeichnung5 ?? String.Empty,
                        MwstSatz = bundleArtikel.MWST_N_PROZENT,
                        Gebinde = bundleArtikel.WK5_ARTI_N_GEBINDE,
                        AnfrageArtikel = bundleArtikel.WK5_ARTI_L_ANFRAGEARTIKEL,
                        Vpe = bundleArtikel.ARTI_N_COLLI,
                        Langtext = bundleArtikel.Langtext ?? String.Empty,
                        ARTI_L_LAGERFUEHR = bundleArtikel.ARTI_L_LAGERFUEHR,
                        PositionBenötigtSeriennummer = bundleArtikel.ARTI_L_SN,
                        IstAbverkaufsartikel = bundleArtikel.ARTI_L_ABVERKAUF,
                        BPOS_A_TYP = EnumHelper.GetBelegTypString(BelegTyp),
                        BPOS_N_NR = Input.Belegnummer,
                        Durchschnittseinkaufspreis = 0,
                        Preis = 0,
                        BundleId = -1,
                        BundleParent = pos,
                        Menge = bundleArtikel.BUNDLE_N_MENGE
                    };

                    pos.BundleArtikel.Add(subPos);

                    Input.Positionen.Add(subPos);
                }
            }

            await CheckForPreisAsync(pos);

            decimal gewicht = Input.GetBelegGewicht();
            // Bundle werden bereits vor der Generierung zu den Positionen hinzugefügt, da ansonsten das Verschieben in der Beleganlage nicht mehr funktioniert für Bundles
            if (!pos.IstBundle)
            {
                Input.Positionen.Add(pos);
            }

            await PositionSelected(pos);

            if (Input.Frachtkosten > 0 && gewicht != Input.GetBelegGewicht() && (BelegTyp is BelegTyp.Angebot or BelegTyp.Auftrag))
            {
                _showVersandNeuBerechnenModal = true;
            }

            if (!String.IsNullOrWhiteSpace(artikel.WK5_ARTI_A_HINWEIS))
            {
                ArtikelHinweisText = artikel.WK5_ARTI_A_HINWEIS;
            }

            return pos;
        }
        protected Task PositionKopierenAsync(Belegposition pos)
        {
            var newPos = new Belegposition()
            {
                Artikelnummer = pos.Artikelnummer,
                Bezeichnung1 = pos.Bezeichnung1,
                Bezeichnung2 = pos.Bezeichnung2,
                Bezeichnung3 = pos.Bezeichnung3,
                Bezeichnung4 = pos.Bezeichnung4,
                Bezeichnung5 = pos.Bezeichnung5,
                Bild = pos.Bild,
                BildId = pos.BildId,
                Gebinde = pos.Gebinde,
                Rabattbezeichnung = pos.Rabattbezeichnung,
                Option = pos.Option,
                Langtext = pos.Langtext,
                LangtextDrucken = pos.LangtextDrucken,
                BPOS_A_KUNDENR = pos.BPOS_A_KUNDENR,
                Menge = pos.Menge,
                MwstSatz = pos.MwstSatz,
                PosOhneDruck = pos.PosOhneDruck,
                Preis = pos.Preis,
                BPOS_N_RABATTWERT1 = pos.BPOS_N_RABATTWERT1,
                BPOS_N_RABATTWERT2 = pos.BPOS_N_RABATTWERT2,
                AnfrageArtikel = pos.AnfrageArtikel,
                PosEinkaufspreis = pos.PosEinkaufspreis,
                PosNr = ++_zähler,
                Gewicht = pos.Gewicht,
                Vpe = pos.Vpe,
                BPOS_A_TYP = EnumHelper.GetBelegTypString(BelegTyp)
            };

            if (pos.IstBundle)
            {
                newPos.BundleId = -1;
                foreach (var bundlePos in pos.BundleArtikel)
                {
                    newPos.BundleArtikel.Add(new Belegposition
                    {
                        Artikelnummer = bundlePos.Artikelnummer,
                        Bezeichnung1 = bundlePos.Bezeichnung1,
                        Bezeichnung2 = bundlePos.Bezeichnung2,
                        Bezeichnung3 = bundlePos.Bezeichnung3,
                        Bezeichnung4 = bundlePos.Bezeichnung4,
                        Bezeichnung5 = bundlePos.Bezeichnung5,
                        Bild = bundlePos.Bild,
                        BildId = bundlePos.BildId,
                        Gebinde = bundlePos.Gebinde,
                        Rabattbezeichnung = bundlePos.Rabattbezeichnung,
                        Option = bundlePos.Option,
                        Langtext = bundlePos.Langtext,
                        LangtextDrucken = bundlePos.LangtextDrucken,
                        BPOS_A_KUNDENR = bundlePos.BPOS_A_KUNDENR,
                        Menge = bundlePos.Menge,
                        MwstSatz = bundlePos.MwstSatz,
                        PosOhneDruck = bundlePos.PosOhneDruck,
                        Preis = bundlePos.Preis,
                        BPOS_N_RABATTWERT1 = bundlePos.BPOS_N_RABATTWERT1,
                        BPOS_N_RABATTWERT2 = bundlePos.BPOS_N_RABATTWERT2,
                        AnfrageArtikel = bundlePos.AnfrageArtikel,
                        PosEinkaufspreis = bundlePos.PosEinkaufspreis,
                        PosNr = ++_zähler,
                        Gewicht = bundlePos.Gewicht,
                        Vpe = bundlePos.Vpe,
                        BPOS_A_TYP = EnumHelper.GetBelegTypString(BelegTyp),
                        BundleParent = newPos,
                        BundleId = -1
                    });
                }
            }

            Input.Positionen.Add(newPos);
            SelectedPosition = newPos;
            return Task.CompletedTask;
        }
        protected Task RemovePositionAsync(Belegposition pos)
        {
            decimal gewicht = Input.GetBelegGewicht();

            // Bei Gutschriften wollen wir ggf. nur einen Teil Gutschreiben. Daher dürfen nicht alle Positionen entfernt werden
            if (pos.IstBundle && BelegTyp is not BelegTyp.Gutschrift)
            {
                foreach (var bundlePos in pos.BundleArtikel)
                {
                    Input.Positionen.Remove(bundlePos);
                    _zähler--;
                }
            }

            Input.Positionen.Remove(pos);
            _zähler--;

            for (int i = 0; i < _zähler; i++)
            {
                Input.Positionen[i].PosNr = i + 1;
            }

            if (Input.Frachtkosten > 0 && gewicht != Input.GetBelegGewicht() && (BelegTyp is BelegTyp.Angebot or BelegTyp.Auftrag))
            {
                _showVersandNeuBerechnenModal = true;
            }

            return PositionSelected(Input.Positionen.FirstOrDefault());
        }
        protected async Task AddArtikelAsync(string artikel)
        {
            using FbController2 fbController = new FbController2();
            Artikel? arti = await artikelService.GetAsync(artikel, fbController);
            if (arti is null)
            {
                await jsRuntime.ShowToast(ToastType.error, $"Der Artikel {artikel} konnte nicht gefunden werden.");
            }
            else
            {
                await AddArtikelAsync(arti);
            }
        }

        protected async Task CheckForPreisAsync(Belegposition pos)
        {
            if (Kunde is null)
            {
                return;
            }

            using FbController2 fbController = new FbController2();
            Artikel? artikel = await artikelService.GetAsync(pos.Artikelnummer, fbController);


            if (artikel is not null)
            {
                // Staffelpreise
                var (preis, rabatt, rabattbezeichnung) = artikel.GetPreis(pos.Menge, Kunde.KUND_N_PREISLISTNR);
                pos.Preis = preis;


                if (rabatt > 0)
                {
                    pos.Rabatt1 = rabatt;
                    pos.Rabattbezeichnung = rabattbezeichnung;
                }
                else
                {
                    // Wir müssen den Rabatt zurücksetzen, da ansonsten ein zuvor eingetragener Rabatt nicht überschrieben wird, wenn kein Rabatt mehr angewendet wird.
                    pos.Rabatt1 = 0;
                    pos.Rabattbezeichnung = string.Empty;
                }



                // Kundenpreise
                Kundenrabatt? kundenpreise = await Kundenrabatt.GetKundenSachnummerAsync(Kunde.KUND_A_NR, pos.Artikelnummer);

                if (kundenpreise is not null)
                {
                    var (rabatt1, _) = kundenpreise.GetRabatte(pos.Menge);

                    // Da wir als erstes den Staffelpreis von Artikel übernehmen, müssen wir prüfen, ob der Staffelpreis des Kundenpreises ggf. höher ist.
                    if (pos.Rabatt1 < rabatt1)
                    {
                        pos.Rabatt1 = rabatt1;
                        pos.Rabattbezeichnung = "Stammkunde";
                        if (rabatt1 > 0)
                        {
                            pos.Rabatt2 = 0;
                        }
                    }
                }

                // Sonderangebote
                List<ArtikelAngebot> sonderangebote = await ArtikelAngebot.GetAngeboteAsync(pos.Artikelnummer).ToListAsync();

                foreach (var sonderangebot in sonderangebote.Where(x => x.ARAN_N_PREISLISTE == Kunde.KUND_N_PREISLISTNR))
                {
                    if ((sonderangebot.SolangeDerVorratReicht && pos.Menge <= artikel.Bestand) || (!sonderangebot.SolangeDerVorratReicht && sonderangebot.ARAN_D_ENDE >= DateTime.Now))
                    {
                        if (sonderangebot.ARAN_N_PREIS < pos.PreisMitRabatt)
                        {
                            pos.Preis = sonderangebot.ARAN_N_PREIS;
                            pos.Rabatt1 = 0;
                            pos.Rabatt2 = 0;
                            pos.Rabattbezeichnung = "Aktion";
                        }
                    }
                }
            }
        }



        protected string GetBedarfPopover()
        {
            string bedarf = "<ul>";
            foreach (int nr in SelectedPositionBedarfsAufträge)
            {
                bedarf += $"<li><a href='/Auftraege/{nr}' target='_blank'>AU-{nr}</a></li>";
            }

            bedarf += "</ul>";
            return bedarf;
        }

        protected string GetBestellPopover()
        {
            string bedarf = "<ul>";
            foreach (int nr in SelectedPositionBestellungen)
            {
                bedarf += $"<li><a href='/Bestellungen/{nr}' target='_blank'>{nr}</a></li>";
            }

            bedarf += "</ul>";
            return bedarf;
        }
        #endregion
        #region IEditPage
        public virtual string ActivePageName { get; set; } = string.Empty;
        public virtual string FinalBreadcrumbItemName { get; set; } = string.Empty;
        public EditMode Modus { get; set; } = EditMode.Anlage;


        public virtual Task LadeEditModeAsync()
        {
            throw new NotImplementedException();
        }
        public string GetNavListClass(bool active) => active ? "nav-link active" : "nav-link";
        public string GetTabClass(bool active) => active ? "tab-pane fade active show" : "tab-pane fade";
        public string GetSubNavListClass(bool active) => active ? "list-group-item list-group-item-action active" : "list-group-item list-group-item-action";
        #endregion
        protected async virtual Task KundeSelectedAsync(Kunde kunde, bool showKundenNotiz = false, bool checkMwst = false)
        {
            Kunde = kunde;
            _ansprechpartner = await Ansprechpartner.GetAnsprechpartnerAsync(kunde.KUND_A_NR).ToListAsync();
            _lieferanschriften = await Lieferanschrift.GetLieferanschriftenAsync(kunde.KUND_A_NR).ToListAsync();
            _rechnungsanschriften = await Rechnungsanschrift.GetRechnungsanschriftenAsync(kunde.KUND_A_NR).ToListAsync();

            var hauptAnsprechpartner = _ansprechpartner.Where(x => x.PART_WK5_L_HAUPT).FirstOrDefault();
            var hauptLieferanschrift = _lieferanschriften.Where(x => x.KULA_WK5_L_HAUPTANSCHRIFT).FirstOrDefault();
            var hauptRechnungsanschrift = _rechnungsanschriften.Where(x => x.KURE_WK5_L_HAUPTANSCHRIFT).FirstOrDefault();

            string anrede = (hauptAnsprechpartner is not null ? hauptAnsprechpartner.PART_A_BRIEFANREDE : kunde.KUND_A_ANREDE) ?? "Sehr geehrte Damen und Herren";

            // Prüfen ob der Kunde im EU-Ausland ist und für Ihn Mehrwertsteuer berechnet wird
            if (checkMwst)
            {
                await CheckForMehrwertsteuerAsync(kunde, hauptLieferanschrift);
            }

            Input.Name1 = kunde.KUND_A_NAME1 ?? String.Empty;
            Input.Name2 = kunde.KUND_A_NAME2 ?? String.Empty;
            Input.Name3 = kunde.KUND_A_NAME3 ?? String.Empty;
            Input.LieferbedingungId = kunde.KUND_N_LIEFERUNG;
            Input.ZahlungsbedingungId = kunde.KUND_N_ZAHLUNG;
            Input.LieferanschriftId = hauptLieferanschrift is not null ? hauptLieferanschrift.KULA_N_NR : 0;
            Input.RechnungsanschriftId = hauptRechnungsanschrift is not null ? hauptRechnungsanschrift.KURE_N_NR : 0;
            Input.Ansprechpartner = hauptAnsprechpartner is not null ? hauptAnsprechpartner.GetName() : String.Empty;
            Input.VertreterId = kunde.KUND_N_VERTRETNUMM;
            Input.Anrede = anrede;
            Input.Land = kunde.KUND_A_LAND;
            Input.Strasse = kunde.KUND_A_STRASSE ?? String.Empty;
            Input.Postleitzahl = kunde.KUND_A_PLZ ?? String.Empty;
            Input.MwstBerechnen = kunde.KUND_L_MWST;
            Input.Ort = kunde.KUND_A_ORT ?? String.Empty;
            Input.NeutralerVersender = kunde.KUND_L_NEUTRALVERSAND;

            _showKundenSucheModal = false;

            if (showKundenNotiz && !String.IsNullOrWhiteSpace(Kunde.NotizAsText))
            {
                _showKundenInfoModal = true;
            }
        }
        protected Task LieferbedingungSelectedAsync(int lieferbedingungId)
        {
            Input.LieferbedingungId = lieferbedingungId;
            SelectedLieferbedingung = Program.AppDaten.Lieferbedingungen.FirstOrDefault(x => x.LIBD_N_NR == lieferbedingungId);
            return Task.CompletedTask;
        }
        protected Task KundenSucheClosedAsync()
        {
            if (Kunde is null)
            {
                navigationManager.NavigateTo(EnumHelper.GetBelegTypUrl(BelegTyp));
            }
            _showKundenSucheModal = false;
            return Task.CompletedTask;
        }
        protected async Task LieferanschriftChangedAsync(int lieferanschriftId, bool checkMwst = false)
        {
            Input.LieferanschriftId = lieferanschriftId;
            Lieferanschrift = _lieferanschriften.Where(x => x.KULA_N_NR == lieferanschriftId).FirstOrDefault();
            NurDurchAdminZuBearbeiten = Lieferanschrift is not null && !Lieferanschrift.KULA_A_LAND.Equals(Input.Land);
            if (checkMwst && Kunde is not null)
            {
                await CheckForMehrwertsteuerAsync(Kunde, Lieferanschrift);
            }
        }
        protected Task RechnungsanschriftChangedAsync(int rechnungsanschriftId)
        {
            Input.RechnungsanschriftId = rechnungsanschriftId;
            Rechnungsanschrift = _rechnungsanschriften.Where(x => x.KURE_N_NR == rechnungsanschriftId).FirstOrDefault();
            return Task.CompletedTask;
        }
        protected Task AnsprechpartnerChangedAsync(string ansprechpartner)
        {
            Ansprechpartner? partner = _ansprechpartner.Where(x => x.GetName().Equals(ansprechpartner, StringComparison.OrdinalIgnoreCase)).FirstOrDefault();

            Input.Ansprechpartner = ansprechpartner;

            if (partner is not null)
            {
                Input.Anrede = partner.PART_A_BRIEFANREDE ?? "Sehr geehrte Damen und Herren";
                if (Input.Belegtyp is "AN" && string.IsNullOrWhiteSpace(Input.AnfrageDurch))
                {
                    Input.AnfrageDurch = partner.GetName();
                }
                else if (Input.Belegtyp is not "AN" && string.IsNullOrWhiteSpace(Input.BestelltDurch))
                {
                    Input.BestelltDurch = partner.GetName();
                }
            }



            return Task.CompletedTask;
        }
        protected async Task CheckForMehrwertsteuerAsync(Kunde kunde, Lieferanschrift? lieferanschrift)
        {
            VatRequest? request = null;
            if (lieferanschrift is not null)
            {
                var sucheLand = Program.AppDaten.Länder.Where(x => x.LAND_A_ID.Equals(lieferanschrift.KULA_A_LAND)).FirstOrDefault();

                if (sucheLand is not null && sucheLand.LAND_A_ID != "DE")
                {

                    if (sucheLand.WK5_LAND_L_ISTEULAND)
                    {
                        if (String.IsNullOrWhiteSpace(kunde.KUND_A_USTIDNR))
                        {
                            Input.MwstBerechnen = true;
                            await jsRuntime.ShowToast(ToastType.error, "Automatische Prüfung der UstId ist nicht möglich, da im Kunden keine UstId angegeben ist.", 0);
                        }
                        else
                        {
                            request = lieferanschrift.ToVatRequest(kunde.KUND_A_USTIDNR);
                        }
                    }
                    else
                    {
                        Input.MwstBerechnen = false;
                        await jsRuntime.ShowToast(ToastType.warning, "Mehrwertsteuer wurde durch Länderprüfung auf Nein gesetzt.", 0);
                    }

                }
                else
                {
                    if (!Input.MwstBerechnen)
                    {
                        Input.MwstBerechnen = true;

                        await jsRuntime.ShowToast(ToastType.warning, "Mehrwertsteuer wurde durch Länderprüfung auf Ja gesetzt.", 0);

                    }
                }
            }
            else
            {
                var sucheLand = Program.AppDaten.Länder.Where(x => x.LAND_A_ID.Equals(kunde.KUND_A_LAND)).FirstOrDefault();
                if (sucheLand is not null && sucheLand.LAND_A_ID != "DE")
                {
                    if (sucheLand.WK5_LAND_L_ISTEULAND)
                    {
                        if (String.IsNullOrWhiteSpace(kunde.KUND_A_USTIDNR))
                        {
                            Input.MwstBerechnen = true;
                            await jsRuntime.ShowToast(ToastType.error, "Automatische Prüfung der UstId ist nicht möglich, da im Kunden keine UstId angegeben ist.", 0);
                        }
                        else
                        {
                            request = kunde.ToVatRequest();
                        }
                    }
                    else
                    {
                        Input.MwstBerechnen = false;
                        await jsRuntime.ShowToast(ToastType.warning, "Mehrwertsteuer wurde durch Länderprüfung auf Nein gesetzt.", 0);
                    }

                }
                else
                {
                    if (!Input.MwstBerechnen)
                    {
                        Input.MwstBerechnen = true;
                        await jsRuntime.ShowToast(ToastType.warning, "Mehrwertsteuer wurde durch Länderprüfung auf Ja gesetzt.", 0);
                    }
                }
            }

            if (request is not null)
            {
                using FbController2 fbController = new FbController2();
                VatCheck result = await vatService.CheckVatAsync(request, fbController);
                bool isValid = vatService.IsVatCheckValid(result);

                Input.MwstBerechnen = !isValid;
                await jsRuntime.ShowToast(ToastType.warning, $"Mehrwertsteuer des Belegs wurde nach UstId Prüfung auf {(Input.MwstBerechnen ? "Ja" : "Nein")} gesetzt.", 0);


                if (lieferanschrift is not null)
                {
                    lieferanschrift.UstIdLastPrüfung = DateTime.Now;
                    lieferanschrift.IstUstIdGültig = isValid;
                }
                else
                {
                    kunde.UstIdLastPrüfung = DateTime.Now;
                    kunde.IstUstIdGültig = isValid;
                    kunde.KUND_L_MWST = !isValid;
                }


            }
        }
        protected async Task LadeArtikelzubehör()
        {
            if (SelectedPosition is not null)
            {
                using FbController2 fbController = new FbController2();
                Zubehör = await ArtikelZubehör.GetArtikelZubehörAsync(SelectedPosition.Artikelnummer, fbController).ToListAsync();
            }
        }
        protected async Task LadeKundenArtikel()
        {
            if (Kunde is not null)
            {
                using FbController2 fbController = new FbController2();
                HistorischeKundenArtikel = await artikelService.GetKundenBestellteArtikel(Kunde.KUND_A_NR, fbController).ToListAsync();
            }
        }
        protected async Task LadeHistorieAsync()
        {
            _historie.Clear();
            _lädtHistorie = true;
            await foreach (var item in BelegChangeAnsicht.GetHistorieAsync(Input.Belegtyp, Input.Belegnummer.ToString()))
            {
                _historie.Add(item);
                StateHasChanged();
            }
            _lädtHistorie = false;
        }
        protected Task ShowKundenKontakte(string kundennummer)
        {
            _selectedKundennummer = kundennummer;
            _showKundenkontakte = true;
            return Task.CompletedTask;
        }
        protected async Task BerechneVersandkostenAsync()
        {
            if (Gesperrt)
            {
                return;
            }
            VersandHinweise.Clear();
            _lädtVersandkosten = true;
            _shippingOptions.Clear();

            List<PaketInput> pakete = new List<PaketInput>();

            decimal restGewicht = Input.GetBelegGewicht(true);

            while (restGewicht > 0)
            {
                decimal paketGewicht = restGewicht > 31 ? 31 : restGewicht;
                pakete.Add(new PaketInput
                {
                    Gewicht = paketGewicht
                });

                restGewicht -= paketGewicht;
            }
            KundenAnschrift? anschrift = null;
            if (Lieferanschrift is not null)
            {
                anschrift = new KundenAnschrift(Lieferanschrift);
            }
            else
            {
                anschrift = new KundenAnschrift(Input);
            }

            GewichtsberechnerDestinationInfoInput desitinationInfoInput = new GewichtsberechnerDestinationInfoInput
            {
                City = anschrift.Ort,
                Country = anschrift.Land,
                ZipCode = anschrift.Postleitzahl
            };

            try
            {
                _shippingOptions = await versandService.GetShippingOptionsAsync(pakete, desitinationInfoInput, Input.GetNettoBetrag(Program.AppDaten.Optionen), Program.AppDaten.Länder, jsRuntime);
            }
            catch (Exception ex)
            {
                _lädtVersandkosten = false;
                await jsRuntime.ShowToast(ToastType.error, $"Fehler beim laden der Versandoptionen. Fehler: {ex.Message}");
                return;
            }
            _lädtVersandkosten = false;

            if (Input.Positionen.Sum(x => x.Preis * x.Menge) >= 1000
                && Program.AppDaten.Länder.Any(x => !x.WK5_LAND_L_ISTEULAND && x.LAND_A_ID == desitinationInfoInput.Country)
                && desitinationInfoInput.Country != "DE")
            {
                VersandHinweise.Add(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "ACHTUNG! +3 Tage Lieferzeit aufgrund von Lieferung in Drittland. Wurde der Kunde darüber informiert?"
                });
            }

            if (!_shippingOptions.Any())
            {
                await jsRuntime.ShowToast(ToastType.error, "Es konnten keine Versandoptionen ermittelt werden.");
            }
        }
        protected async Task CheckÜbernahmeKundeAsync(Beleg beleg, Kunde kunde)
        {
            if (!beleg.Name1.Equals(kunde.KUND_A_NAME1))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Name 1 des Kunden hat sich geändert"
                });
            }

            if (!beleg.Name2.Equals(kunde.KUND_A_NAME2))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Name 2 des Kunden hat sich geändert"
                });
            }

            if (!beleg.Name3.Equals(kunde.KUND_A_NAME3))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Name 3 des Kunden hat sich geändert"
                });
            }

            if (!beleg.Land.Equals(kunde.KUND_A_LAND))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Land des Kunden hat sich geändert"
                });
            }

            if (!beleg.Strasse.Equals(kunde.KUND_A_STRASSE))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Straße des Kunden hat sich geändert"
                });
            }

            if (!beleg.Postleitzahl.Equals(kunde.KUND_A_PLZ))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Postleitzahl des Kunden hat sich geändert"
                });
            }

            if (!beleg.Ort.Equals(kunde.KUND_A_ORT))
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Ort des Kunden hat sich geändert"
                });
            }
        }
        protected async Task LadeMwstNeuAsync()
        {
            //using FbController2 fbController = new FbController2();
            Kunde? kunde = await Kunde.GetKundeAsync(Input.Kundennummer);

            if (kunde is not null)
            {
                if (Input.MwstBerechnen != kunde.KUND_L_MWST)
                {
                    Input.MwstBerechnen = kunde.KUND_L_MWST;
                    await CheckForMehrwertsteuerAsync(kunde, Lieferanschrift);
                }
                else
                {
                    await jsRuntime.ShowToast(ToastType.success, "Berechnung der Mehrwertsteuer wurde erfolgreich aktualisiert. Es wurde keine Differenz festgestellt.", 0);
                }
            }
        }
        #region Abstrakte Funktionen
        public abstract Task DownloadAsync(bool confirmed = false);
        public abstract Task ÖffneMailAsync(bool confirmed = false);
        #endregion        

        public override void AddHotKeys(HotKeysContext context)
        {
            context
            .Add(ModKeys.Alt, Keys.Insert, HandleCtrlPlus, "Alias für das Plus Symbol", Exclude.Default)
            .Add(ModKeys.Alt, Keys.Right, NextTab, "Wechselt zum nächsten Tab", Exclude.Default)
            .Add(ModKeys.Alt, Keys.Left, PrevTab, "Wechselt zum vorherigen Tab", Exclude.Default)
            .Add(ModKeys.Alt, Keys.P, async () =>
             {
                 if (Modus is EditMode.Bearbeiten)
                 {
                     await OpenPreviewWindow();
                 }
             }, "Öffnet die Druckvorschau", Exclude.Default)
            .Add(ModKeys.Alt, Keys.D, async () =>
            {
                if (Modus is EditMode.Bearbeiten)
                {
                    await DownloadAsync();
                    StateHasChanged();
                }
            }, "Lädt das Angebot runter", Exclude.Default)
            .Add(ModKeys.Alt, Keys.E, async () =>
            {
                if (Modus is EditMode.Bearbeiten)
                {
                    await ÖffneMailAsync();
                    StateHasChanged();
                }
            }, "Startet den Versand per Email", Exclude.Default)
            .Add(ModKeys.Alt, Keys.Z, () =>
            {
                if (Modus is EditMode.Bearbeiten)
                {
                    _showZeiterfassungModal = true;
                    StateHasChanged();
                }
            }, "Öffnet die Zeiterfassung", Exclude.Default)
            .Add(ModKeys.Alt, Keys.T, async () =>
            {
                if (ActiveTab is AngeboteTabs.Positionen or AufträgeTabs.Positionen or LieferscheinTabs.Positionen or RechnungTabs.Positionen or GutschriftenTabs.Positionen)
                {
                    await AddTextAsync();
                    StateHasChanged();
                }
            }, "Fügt eine Textposition hinzu", Exclude.Default)
            .Add(ModKeys.Alt, Keys.PgUp, async () =>
            {
                if (ActiveTab is AngeboteTabs.Positionen or AufträgeTabs.Positionen or LieferscheinTabs.Positionen or RechnungTabs.Positionen or GutschriftenTabs.Positionen && !Gesperrt)
                {
                    await Move(true);
                    StateHasChanged();
                }
            }, "Bewegt die aktuelle Position einen nach oben", Exclude.Default)
            .Add(ModKeys.Alt, Keys.PgDown, async () =>
            {
                if (ActiveTab is AngeboteTabs.Positionen or AufträgeTabs.Positionen or LieferscheinTabs.Positionen or RechnungTabs.Positionen or GutschriftenTabs.Positionen && !Gesperrt)
                {
                    await Move(false);
                    StateHasChanged();
                }
            }, "Bewegt die aktuelle Position einen nach unten", Exclude.Default)
            .Add(ModKeys.None, Keys.PgUp, async () =>
            {
                if (ActiveTab is AngeboteTabs.Positionen or AufträgeTabs.Positionen or LieferscheinTabs.Positionen or RechnungTabs.Positionen or GutschriftenTabs.Positionen)
                {
                    await Select(true);
                    StateHasChanged();
                }
            }, "Wählt die nächste Position aus", Exclude.Default)
            .Add(ModKeys.None, Keys.PgDown, async () =>
            {
                if (ActiveTab is AngeboteTabs.Positionen or AufträgeTabs.Positionen or LieferscheinTabs.Positionen or RechnungTabs.Positionen or GutschriftenTabs.Positionen)
                {
                    await Select(false);
                    StateHasChanged();
                }
            }, "Bewegt die aktuelle Position einen nach oben aus", Exclude.Default)
            .Add(ModKeys.Alt, Keys.K, () =>
            {
                if (!Gesperrt)
                {
                    _showKundenSucheModal = true;
                    StateHasChanged();
                }
            }, "Ändert den Kunden", Exclude.Default)
            .Add(ModKeys.Alt, Keys.I, () =>
            {
                if (!String.IsNullOrWhiteSpace(Kunde?.NotizAsText))
                {
                    _showKundenInfoModal = true;
                    StateHasChanged();
                }
            }, "Zeigt Kundeninformationen an", Exclude.Default)
            ;
        }



        public Task NextTab()
        {
            bool found = false;
            int tabIndex = (Convert.ToInt32(ActiveTab) + 1) % Enum.GetValues(typeof(TEnumTabs)).Length;
            while (!found)
            {
                TEnumTabs tab = (TEnumTabs)(object)tabIndex;
                if (EditOnlyTabs.Contains(tab) && Modus != EditMode.Bearbeiten)
                {
                    tabIndex = (tabIndex + 1) % Enum.GetValues(typeof(TEnumTabs)).Length;
                    continue;
                }

                if (AdminOnlyTabs.Contains(tab) && !Mitarbeiter.IsAdmin)
                {
                    tabIndex = (tabIndex + 1) % Enum.GetValues(typeof(TEnumTabs)).Length;
                    continue;
                }

                found = true;
            }

            ActiveTab = (TEnumTabs)(object)tabIndex;
            StateHasChanged();
            return Task.CompletedTask;
        }

        public Task PrevTab()
        {
            bool found = false;
            int tabIndex = Convert.ToInt32(ActiveTab) - 1 < 0 ? Enum.GetValues(typeof(TEnumTabs)).Length - 1 : Convert.ToInt32(ActiveTab) - 1;
            while (!found)
            {
                TEnumTabs tab = (TEnumTabs)(object)tabIndex;
                if (EditOnlyTabs.Contains(tab) && Modus != EditMode.Bearbeiten)
                {
                    tabIndex = tabIndex - 1 < 0 ? Enum.GetValues(typeof(TEnumTabs)).Length - 1 : tabIndex - 1;
                    continue;
                }

                if (AdminOnlyTabs.Contains(tab) && !Mitarbeiter.IsAdmin)
                {
                    tabIndex = tabIndex - 1 < 0 ? Enum.GetValues(typeof(TEnumTabs)).Length - 1 : tabIndex - 1;
                    continue;
                }

                found = true;
            }

            ActiveTab = (TEnumTabs)(object)tabIndex;
            StateHasChanged();
            return Task.CompletedTask;
        }

        public Task HandleCtrlPlus()
        {
            switch (ActiveTab)
            {
                case AngeboteTabs.Positionen or AufträgeTabs.Positionen or LieferscheinTabs.Positionen or RechnungTabs.Positionen or GutschriftenTabs.Positionen:
                    if (!Gesperrt)
                    {
                        _showArtikelSucheModal = true;
                        StateHasChanged();
                    }
                    break;
                default:
                    break;
            }
            return Task.CompletedTask;
        }

        public async Task Move(bool moveUp)
        {
            if (SelectedPosition is not null)
            {
                if (moveUp && CanMoveUp(SelectedPosition))
                {
                    await MovePosition(SelectedPosition, moveUp);
                }
                else if (!moveUp && CanMoveDown(SelectedPosition))
                {
                    await MovePosition(SelectedPosition, moveUp);
                }
            }
        }

        public Task Select(bool moveUp)
        {
            if (SelectedPosition is not null)
            {
                int max = Input.Positionen.Last().PosNr;
                int newPos = moveUp ? SelectedPosition.PosNr - 1 : SelectedPosition.PosNr + 1;
                if (moveUp)
                {

                    if (newPos < 1)
                    {
                        newPos = max;
                    }
                }
                else
                {
                    if (newPos > max)
                    {
                        newPos = 1;
                    }

                }

                var pos = Input.Positionen.FirstOrDefault(x => x.PosNr == newPos);

                if (pos is not null)
                {
                    return PositionSelected(pos);
                }
                else
                {
                    return PositionSelected(Input.Positionen.FirstOrDefault());
                }
            }

            return Task.CompletedTask;
        }

        protected async Task OpenPreviewWindow()
        {
            await jsRuntime.InvokeAsync<object>("open", new object[] { $"/Verkauf/Vorschau/{BelegTyp}/{Input.Belegnummer}", $"popout-{Input.Belegnummer}", "scrollbars=no,resizable=no,status=no,location=no,toolbar=no,menubar=no,width=0,height=0,left=-1000,top=-1000" });
        }

        protected Task LieferterminFocusAsync()
        {
            if (Input.Liefertermin == default)
            {
                Input.Liefertermin = DateTime.Now;
            }
            return Task.CompletedTask;
        }
    }
}
