﻿using KarleyLibrary.Erweiterungen;
using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Forms;
using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using Toolbelt.Blazor.HotKeys;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Drucken.Aufträge;
using WK5.Core.Email;
using WK5.Core.Models;
using WK5.Core.PageModels.Verkauf.Aufträge;
using WK5.Core.Services;

namespace WK5_Blazor.Pages.Verkauf.Aufträge
{
    public partial class AuftragsForm
    {
        [Parameter] public int Auftragsnummer { get; set; }
        [Parameter] public int RmaNummer { get; set; }

        public Pipeline<Auftrag> SubmitPipeline { get; set; } = new Pipeline<Auftrag>();
        public Pipeline<Auftrag> ÜbernahmePipeline { get; set; } = new Pipeline<Auftrag>();
        public Pipeline<Auftrag> FreigabePipeline { get; set; } = new Pipeline<Auftrag>();

        public List<int> Bestätigungsnummern { get; set; } = new List<int>();

        private bool _showKundenArtikelhistorieModal;
        private bool _showErledigtModal;
        private bool _showPausierenModal;
        private bool _showÜbernahmeModal;
        private Func<bool, Task>? _confirmAction;
        private bool _confirmIsMail = false;
        private bool _showArtikelzubehör = false;
        private bool _showHistorischeKundenArtikel = false;
        private bool _showZahlungNichtErhaltenModal;
        private bool _showLieferdatumNichtErreichtModal;
        private bool _showTeillieferungsFreigabeModal;
        private bool _showKreditLimitÜberschrittenModal = false;
        private bool _showDoppelterVersandModal = false;
        private bool _showNeutralerLieferscheinUploadModal;

        private decimal _freiesKreditlimit = 0.0m;
        private List<Bestellung> _bestellungen = new List<Bestellung>();
        private Rma? _rma;

        public AuftragsForm()
        {
            Input = new Auftrag()
            {
                MwstBerechnen = true
            };

            FilterPakete = new PaketFilter
            {
                Typ = PaketTyp.Auftrag
            };

            FilterZeiterfassung.Option = ZeitenFilterOption.Auftrag;
            StartCopy = new Auftrag();
            ActiveTab = AufträgeTabs.Kundendaten;
            EditOnlyTabs.Add(AufträgeTabs.Dokumentenablage);
            EditOnlyTabs.Add(AufträgeTabs.Historie);
        }
        public override string ActivePageName => Modus switch
        {
            EditMode.Anlage => "Neuer Auftrag",
            EditMode.Bearbeiten => $"AU-{Auftragsnummer}",
            _ => "Unbekannt"
        };
        public override string FinalBreadcrumbItemName => Modus switch
        {
            EditMode.Anlage => "Neu",
            EditMode.Bearbeiten => $"{Auftragsnummer}",
            _ => "Unbekannt"
        };
        protected override async Task OnParametersSetAsync()
        {
            _versandMehrwertsteuer = await Mehrwertsteuer.GetVersandMehrwertsteuerAsync();

            if (Auftragsnummer > 0)
            {
                await LadeEditModeAsync();
            }
            FilterPakete.Nummer = Auftragsnummer.ToString();
            FilterZeiterfassung.Suchbegriff = Auftragsnummer.ToString();
            FilterZeiterfassung.Kundennummer = Input.Kundennummer;
            StartCopy = Input.DeepCopy();

            if (Modus is EditMode.Anlage)
            {
                using FbController2 fbController = new FbController2();
                if (RmaNummer is > 0)
                {
                    _rma = await rmaService.GetAsync(RmaNummer, fbController);

                    if (_rma is not null)
                    {
                        Input.RmaNummerVerknüpfung = RmaNummer;

                        Kunde? kunde = await Kunde.GetKundeAsync(_rma.Kundennummer);
                        if (kunde is not null)
                        {
                            await KundeSelectedAsync(kunde, true, true);
                        }
                    }
                }


            }

            if (Modus is EditMode.Anlage && Mitarbeiter.IsTechnik)
            {
                Input.IstTechnikbeleg = true;
            }


            if (Kunde is null)
            {
                _showKundenSucheModal = true;
            }



            await SetupSubmitPipelineAsync();
            await SetupFreigabePipelineAsync();
            await SetupÜbernahmePipelineAsync();
            await SetAlerts();
        }
        public async override Task LadeEditModeAsync()
        {
            using FbController2 fbController = new FbController2();
            Auftrag? auftrag = await Auftrag.GetAuftragAsync(Auftragsnummer, fbController);
            if (auftrag is not null)
            {
                Kunde = await Kunde.GetKundeAsync(auftrag.Kundennummer);
                if (Kunde is not null)
                {
                    await KundeSelectedAsync(Kunde);
                    Input = auftrag;
                    _zähler = auftrag.Positionen.Count;
                    Gesperrt = auftrag.BELE_L_UBERNAHME || auftrag.LieferscheinnummerVerknüpfung > 0;
                    GesperrtDurchZahlung = auftrag.BELE_L_BEZAHLT;
                    await LieferanschriftChangedAsync(Input.LieferanschriftId);
                    await RechnungsanschriftChangedAsync(Input.RechnungsanschriftId);
                    _bestellungen = await auftrag.GetBestellungenAsync(fbController).ToListAsync();
                    await PositionSelected(Input.Positionen.FirstOrDefault());
                    await LieferbedingungSelectedAsync(auftrag.LieferbedingungId);
                    _subBelege = await belegService.GetSubBelege(BelegTyp.Auftrag, Auftragsnummer, fbController);
                    Modus = EditMode.Bearbeiten;
                    await LadeKundenArtikel();
                }

                if (auftrag.WK5_BELE_L_LIMITPRUEFUNG)
                {
                    await AddAlertAsync(new AlertBox
                    {
                        AlertType = AlertType.Warning,
                        Message = $"Die Überschreitung des Kreditlimit muss für den Auftrag noch freigegeben werden"
                    });
                }
            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Auftrag {Auftragsnummer} konnte nicht gefunden werden."
                });


                navigationManager.NavigateTo("/Auftraege");
            }


            if (!_isDisposed)
            {
                await CheckActivePageAsync();
            }

        }
        public async Task SetupSubmitPipelineAsync()
        {
            SubmitPipeline = new Pipeline<Auftrag>();
            await SubmitPipeline.HardReset();
            // Prüfen ob es für den Kunden schon einen Auftrag mit der selben Bestätigungsnummer gibt
            SubmitPipeline.Add(async (input) =>
            {
                if (String.IsNullOrWhiteSpace(input.Bestellnummer))
                {
                    return true;
                }

                if (StartCopy.Bestellnummer.Equals(input.Bestellnummer, StringComparison.OrdinalIgnoreCase))
                {
                    return true;
                }

                //TODO (BIG): Logik in BelegService integrieren
                string sql = "SELECT BELE_N_NR FROM BELEGE WHERE BELE_A_TYP = 'AU' AND BELE_A_BESTAET_NR = @BESTAETNR AND BELE_A_KUNDENNR = @KUNDENNR AND BELE_N_NR != @BELEGNUMMER";

                using FbController2 fbController = new FbController2();
                fbController.AddParameter("@BESTAETNR", input.Bestellnummer);
                fbController.AddParameter("@KUNDENNR", input.Kundennummer);
                fbController.AddParameter("@BELEGNUMMER", input.Belegnummer);
                DataTable data = await fbController.SelectDataAsync(sql);

                if (data.Rows.Count > 0)
                {
                    foreach (DataRow row in data.Rows)
                    {
                        int belegnummer = row.Field<int>("BELE_N_NR");
                        if (belegnummer > 0)
                        {
                            Bestätigungsnummern.Add(belegnummer);
                        }
                    }
                    return false;
                }
                else
                {
                    return true;
                }
            });


            SubmitPipeline.Add(async (input) =>
            {
                if (Modus != EditMode.Anlage)
                {
                    return true;
                }

                using FbController2 fbController = new FbController2();
                fbController.AddParameter("@KUNDENNR", Kunde.KUND_A_NR);
                return true;
            });

            SubmitPipeline.Add(async (input) =>
            {
                await SubmitAsync();
                return true;
            });
        }
        public async Task SetupFreigabePipelineAsync()
        {
            await FreigabePipeline.HardReset();

            FreigabePipeline.Add((input) =>
            {
                bool zahlungErhalten = true;
                bool zahlartIstRechnung = Program.AppDaten.Zahlungsbedingungen.IstRechnung(input.ZahlungsbedingungId) || Program.AppDaten.Zahlungsbedingungen.IstLastschrift(input.ZahlungsbedingungId);

                if (Program.AppDaten.Zahlungsbedingungen.IstVorkasse(input.ZahlungsbedingungId) && !input.BELE_L_BEZAHLT)
                {
                    zahlungErhalten = false;
                }

                if (!zahlungErhalten)
                {
                    _showZahlungNichtErhaltenModal = true;
                    return false;
                }

                return true;
            });

            FreigabePipeline.Add((input) =>
            {
                if (input.Liefertermin != default && input.Liefertermin > DateTime.Now)
                {
                    _showLieferdatumNichtErreichtModal = true;
                    return false;
                }

                return true;
            });

            FreigabePipeline.Add((input) =>
            {
                if (!input.Lieferbar)
                {
                    _showTeillieferungsFreigabeModal = true;
                    return false;
                }

                return true;
            });

            FreigabePipeline.Add(async (input) =>
            {
                using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
                AuftragsfreigabeService auftragsfreigabeService = new AuftragsfreigabeService();
                await auftragsfreigabeService.AuftragFreigebenAysnc(input.Belegnummer, DateTime.Now, fbController);
                await jsRuntime.ShowToast(ToastType.success, $"Der Auftrag wurde freigegeben");
                return true;
            });
        }
        public async Task SetupÜbernahmePipelineAsync()
        {
            await ÜbernahmePipeline.HardReset();

            ÜbernahmePipeline.Add(async (input) =>
            {
                if (Input.Positionen.Any(x => x.ARTI_L_MIETE && x.BPOS_D_MIETE_START.Date < DateTime.Today.Date))
                {
                    await jsRuntime.ShowToast(ToastType.error, "Übernahme fehlgeschlagen, da mindestens ein Mietzeitraum fehlerhaft ist.");
                    return false;
                }

                return true;
            });

            ÜbernahmePipeline.Add(async (input) =>
            {
                await ÜbernahmeAsync();
                return true;
            });
        }
        private async Task SubmitFormAsync()
        {

            if (!NurDurchAdminZuBearbeiten || (NurDurchAdminZuBearbeiten && Mitarbeiter.IsAdmin))
            {
                if (!SperreDurchAnderenUser && Input.HasBeenModified(StartCopy))
                {

                    if (_editForm is null || _editForm.EditContext is null || Kunde is null)
                    {
                        return;
                    }

                    if (_editForm.EditContext.Validate())
                    {
                        SubmitPipeline.Reset();
                        await SubmitPipeline.RunUntilFailureAsync(Input);
                    }
                }
            }



        }
        private async Task SubmitAsync(bool aufErledigt = false)
        {

            if (_editForm is null || _editForm.EditContext is null || Kunde is null)
            {
                return;
            }

            
            if (_editForm.EditContext.Validate())
            {
                using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
                await fbController.StartTransactionAsync();
                try
                {
                    if (Modus is EditMode.Anlage)
                    {
                        await belegService.CreateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);

                        await AddAlertAsync(new AlertBox
                        {
                            AlertType = AlertType.Success,
                            Message = $"Auftrag {Input.Belegnummer} erfolgreich angelegt",
                            DecayTime = 10
                        });
                    }
                    else
                    {
                        if (Input.BELE_L_ERLEDIGT || ((Gesperrt || GesperrtDurchZahlung) && !Mitarbeiter.IsAdmin))
                        {

                            await belegService.UpdateGesperrtenBelegAsync(Input, fbController);

                            await AddAlertAsync(new AlertBox
                            {
                                AlertType = AlertType.Success,
                                Message = $"Notiz, Liefertermin, Letzter Abruf und Einkaufspreise wurden erfolgreich gespeichert",
                                DecayTime = 10
                            });
                        }
                        else if (!Gesperrt || Mitarbeiter.IsAdmin)
                        {
                            await belegService.UpdateAsync(Input, Kunde, Program.AppDaten.Optionen, fbController);

                            await AddAlertAsync(new AlertBox
                            {
                                AlertType = AlertType.Success,
                                Message = $"Auftrag {Input.Belegnummer} erfolgreich gespeichert",
                                DecayTime = 10
                            });
                        }

                        if (aufErledigt)
                        {
                            await belegService.AuftragAufErledigtSetzenAsync(Input, Mitarbeiter.PersonalNummer, fbController);
                        }
                    }
                    await fbController.CommitChangesAsync();
                }
                catch (Exception ex)
                {
                    Debug.WriteLine(ex);
                    await fbController.RollbackChangesAsync();
                    throw;
                }

                if (Modus is EditMode.Anlage)
                {
                    navigationManager.NavigateTo($"/Auftraege/{Input.Belegnummer}");
                }
                else
                {
                    await OnParametersSetAsync();
                }


            }


        }
        private async Task ÜbernahmeAsync()
        {

            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await fbController.StartTransactionAsync();
            try
            {
                Auftrag? auftrag = await Auftrag.GetAuftragAsync(Auftragsnummer, fbController);

                if (auftrag is not null && Kunde is not null)
                {
                    await CheckÜbernahmeKundeAsync(auftrag, Kunde);
                    Lieferschein neuerLieferschein = (Lieferschein)await belegService.ÜbernahmeAsync(auftrag, Kunde, Program.AppDaten.Optionen, Program.AppDaten.VersandMehrwertsteuer, fbController, Mitarbeiter.PersonalNummer);

                    await AddAlertAsync(new AlertBox
                    {
                        AlertType = AlertType.Success,
                        Message = $"AU-{Auftragsnummer} wurde erfolgreich in LS-{neuerLieferschein.Belegnummer} übernommen",
                        DecayTime = 10
                    });

                    await fbController.CommitChangesAsync();
                    navigationManager.NavigateTo($"/Lieferscheine/{neuerLieferschein.Belegnummer}");
                }
            }
            catch (Exception)
            {
                await fbController.RollbackChangesAsync();
                throw;
            }



        }
        private async Task KopierenAsync()
        {
            if (Modus is EditMode.Bearbeiten)
            {
                using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
                await fbController.StartTransactionAsync();
                try
                {

                    Auftrag? auftrag = await Auftrag.GetAuftragAsync(Auftragsnummer, fbController);
                    if (auftrag is null)
                    {
                        throw new ArgumentNullException(nameof(auftrag));
                    }

                    if (Kunde is null)
                    {
                        throw new ArgumentNullException(nameof(Kunde));
                    }

                    var (NeuerBeleg, Messages) = await belegService.CopyAsync(auftrag, Kunde, Program.AppDaten.Optionen, fbController, Mitarbeiter.PersonalNummer);
                    await AddAlertsAsync(Messages);
                    if (NeuerBeleg is not null)
                    {
                        await fbController.CommitChangesAsync();
                        await AddAlertAsync(new AlertBox
                        {
                            AlertType = AlertType.Success,
                            Message = "Auftrag wurde erfolgreich kopiert.",
                            DecayTime = 10
                        });

                        navigationManager.NavigateTo($"/Auftraege/{NeuerBeleg.Belegnummer}");
                    }
                }
                catch (Exception)
                {
                    await fbController.RollbackChangesAsync();
                    throw;
                }


            }
        }
        private async Task ChangePausiertAsync()
        {
            Input.WK5_BELE_L_PAUSIERT = !Input.WK5_BELE_L_PAUSIERT;
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await belegService.AuftragPausierenAsync(Auftragsnummer, Input.WK5_BELE_L_PAUSIERT, fbController);
            await jsRuntime.ShowToast(ToastType.success, Input.WK5_BELE_L_PAUSIERT ? "Auftrag wurde erfolgreich pausiert" : "Auftrag wird nicht mehr pausiert");
            StateHasChanged();
        }
        public async Task ConfirmBestätigungsnummern()
        {
            Bestätigungsnummern.Clear();
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await BelegChange.InsertBelegChangeAsync(fbController, new BelegChange()
            {
                BCNG_A_BELEGTYP = "AU",
                BCNG_N_BELEGNR = Input.Belegnummer.ToString(),
                BCNG_N_ART = 906,
                BCNG_A_WERTALT = "Bestätigt durch",
                BCNG_A_WERTNEU = Mitarbeiter.DisplayName,
            });
            await SubmitPipeline.RunUntilFailureAsync(Input);
        }
        private async Task Auftragfreigeben()
        {
            FreigabePipeline.Reset();
            await FreigabePipeline.RunUntilFailureAsync(Input);
        }
        private async Task TeillieferungsFreigabe()
        {
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);

            //Belegposition pos = new Belegposition();
            //pos.BPOS_N_NR = Input.Belegnummer;
            //pos.BPOS_A_TYP = Input.Belegtyp;
            //pos.Artikelnummer = Belegposition.ARTIKELNR_TEXT;
            //pos.Langtext = $"Von {Mitarbeiter.Username} zur Teillieferung freigegeben";
            //pos.LangtextDrucken = false;
            //pos.PosOhneDruck = true;

            //await belegService.CreatePositionAsync(pos, BelegTyp.Auftrag, true, fbController);
            await FreigabePipeline.RunUntilFailureAsync(Input);
        }



        private async Task CheckRechnungsdruckAsync(Kunde kunde)
        {

            var rechnungsdruckPosition = Input.Positionen.FirstOrDefault(x => x.Artikelnummer == "RECHNUNGSDRUCK");

            if (kunde.KUND_WK5_L_REDRUCK)
            {
                if (rechnungsdruckPosition is null)
                {
                    using FbController2 fbController = new FbController2();
                    Artikel artikel = await artikelService.GetAsync("RECHNUNGSDRUCK", fbController) ?? throw new NullReferenceException(nameof(artikel));
                    var pos = await AddArtikelAsync(artikel);

                    if (pos is not null)
                    {
                        pos.Menge = 1;
                        await jsRuntime.ShowToast(ToastType.warning, "Dem Auftrag wurde die Position Rechnungsdruck hinzugefügt, da der Kunde die Rechnung per Post erhalten möchte.");
                    }

                }
            }
            else
            {
                if (rechnungsdruckPosition is not null)
                {
                    Input.Positionen.Remove(rechnungsdruckPosition);
                    await jsRuntime.ShowToast(ToastType.warning, "Die Position Rechnungsdruck wurde gelöscht, da der Kunde keine Rechnung explizit per Post erhalten möchte.");
                }
            }
        }
        public async override Task DownloadAsync(bool confirmed = false)
        {
            if (Kunde is null)
            {
                return;
            }
            if (!confirmed && Input.Frachtkosten is 0 && !Input.Positionen.Where(x => x.Artikelnummer.Equals("VERSAND")).Any())
            {
                _confirmAction = DownloadAsync;
                _confirmIsMail = false;
                return;
            }
            _isDownloading = true;
            StateHasChanged(); // Wird für den Shortcut benötigt
            using FbController2 fbController = new FbController2();
            var auftrag = await Auftrag.GetAuftragAsync(Auftragsnummer, fbController);

            if (auftrag is not null)
            {

                PrintAuftragRegelsatz printRegeln = new PrintAuftragRegelsatz
                {
                    ShowFooter = true,
                    ShowHeader = true
                };

                if (Program.AppDaten.VersandMehrwertsteuer is null)
                {
                    throw new ArgumentNullException(nameof(Program.AppDaten.VersandMehrwertsteuer));
                }

                PrintAuftrag printer = await PrintAuftrag.CreateAsync(auftrag, printRegeln, fbController);
                string filenameAngebotPdf = printer.Print(GlobalConfig.Configuration.OutputPfad);
                await belegService.SetDruckdatumAsync(BelegTyp.Auftrag, Auftragsnummer, fbController);

                await downloadService.DownloadFile($"AU-{Auftragsnummer}.pdf", await File.ReadAllBytesAsync(filenameAngebotPdf), "application/pdf");
                await downloadService.ClearBuffers();
            }

            _isDownloading = false;
        }
        public async override Task ÖffneMailAsync(bool confirmed = false)
        {

            if (Kunde is null)
            {
                return;
            }

            if (!confirmed && Input.Frachtkosten is 0 && !Input.Positionen.Where(x => x.Artikelnummer.Equals("VERSAND")).Any())
            {
                _confirmAction = ÖffneMailAsync;
                _confirmIsMail = true;
                return;
            }



            _mailIsLoading = true;
            StateHasChanged(); // Wird für den Shortcut benötigt
            using FbController2 fbController = new FbController2();
            var auftrag = await Auftrag.GetAuftragAsync(Auftragsnummer, fbController);

            if (auftrag is not null)
            {
                Ansprechpartner? ansprechpartner = await Ansprechpartner.GetAnsprechpartnerByNameAsync(auftrag.Kundennummer, auftrag.Ansprechpartner);
                string empfängerEmail = Kunde.KUND_A_EMAIL ?? String.Empty;
                if (ansprechpartner is not null && !String.IsNullOrWhiteSpace(ansprechpartner.PART_A_EMAIL))
                {
                    empfängerEmail = ansprechpartner.PART_A_EMAIL ?? String.Empty;
                }


                if (String.IsNullOrWhiteSpace(empfängerEmail))
                {
                    await jsRuntime.ShowToast(ToastType.error, "Auftrag konnte nicht in David geöffnet werden, da beim Kunden keine Mailadresse hinterlegt ist");
                    return;
                }

                KarleyBrowserInterfaceEmail email = new KarleyBrowserInterfaceEmail
                {
                    Absender = GlobalConfig.EmailInfoEU,
                    Empfänger = empfängerEmail,
                    Content = String.Empty,
                    Betreff = $"Auftragsbestätigung {Auftragsnummer}"
                };

                PrintAuftragRegelsatz printRegeln = new PrintAuftragRegelsatz
                {
                    ShowFooter = true,
                    ShowHeader = true
                };

                if (Program.AppDaten.VersandMehrwertsteuer is null)
                {
                    throw new ArgumentNullException(nameof(Program.AppDaten.VersandMehrwertsteuer));
                }

                PrintAuftrag printer = await PrintAuftrag.CreateAsync(auftrag, printRegeln, fbController);
                string filenamePdf = printer.Print(GlobalConfig.Configuration.OutputPfad);
                await belegService.SetDruckdatumAsync(BelegTyp.Auftrag, Auftragsnummer, fbController);
                email.Anhänge.Add(filenamePdf);

                List<KeyValuePair<string, string>> warengruppenDokumente = new List<KeyValuePair<string, string>>()
                {
                    new KeyValuePair<string, string>( "2172", @"\\srv01\daten\Preislisten\_Karley\KARLEY-PREISLISTE-ETIKETTEN.pdf" ), //EtikettenMaterial                    
                };

                List<KeyValuePair<string, string>> unterwarengruppenDokumente = new List<KeyValuePair<string, string>>()
                {
                    new KeyValuePair<string, string>( "2543", @"\\srv01\daten\Preislisten\_Karley\KARLEY-PREISLISTE-BONROLLEN.pdf" ), //Bonrollen                    
                };



                foreach (Belegposition pos in auftrag.GetEndPositionen())
                {
                    IEnumerable<string> wgDoks = warengruppenDokumente.Where(x => x.Key == pos.ARTI_A_WARENGRUPPE).Select(x => x.Value);
                    IEnumerable<string> uwgDoks = unterwarengruppenDokumente.Where(x => x.Key == pos.ARTI_A_UNTERWARENG).Select(x => x.Value);

                    foreach (string dok in wgDoks)
                    {
                        if (!email.Anhänge.Contains(dok))
                        {
                            email.Anhänge.Add(dok);
                        }
                    }

                    foreach (string dok in uwgDoks)
                    {
                        if (!email.Anhänge.Contains(dok))
                        {
                            email.Anhänge.Add(dok);
                        }
                    }
                }


                var (success, data, _) = KarleyLibrary.Serialization.XMLWriter.Serialize(email);
                if (success)
                {
                    string xmlFilename = $"{email.GetHashCode()}.xml";
                    await File.WriteAllTextAsync(Path.Combine(GlobalConfig.Configuration.OutputPfad, xmlFilename), data);
                    await jsRuntime.OpenNewTab($"karley:davidMail;{xmlFilename}");
                }


            }
            _mailIsLoading = false;

        }
        private async Task KreditlimitÜberschritten()
        {
            Input.WK5_BELE_L_LIMITPRUEFUNG = true;
            await SubmitPipeline.RunUntilFailureAsync(Input);
        }
        private Task NeutralerVersandChangedAsync(bool neuerWert)
        {
            Input.NeutralerVersender = neuerWert;
            _showNeutralerLieferscheinUploadModal = neuerWert && Input.NeutralerLieferschein.Length is 0;
            return Task.CompletedTask;
        }
        public override async Task CheckActivePageAsync()
        {
            var (success, page) = ActivePages.AddActivePage(new ActivePage(PageType.Auftrag, Auftragsnummer.ToString(), Mitarbeiter.PersonalNummer));

            if (success)
            {
                Page = page;
            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Der Auftrag wird zurzeit durch {Program.AppDaten.GetMitarbeiterName(page.UserId)} bearbeitet."
                });

                SperreDurchAnderenUser = true;
            }
        }

        private IEnumerable<AlertBox?> GetAlerts()
        {
            if (Input.WK5_BELE_L_PAUSIERT && !Input.BELE_L_STORNO)
            {
                yield return new AlertBox
                {
                    AlertType = AlertType.Info,
                    Message = "Der Auftrag ist aktuell <strong>pausiert</strong>"
                };
            }


            if (NurDurchAdminZuBearbeiten)
            {
                yield return new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Der Auftrag darf nur von einem Admin bearbeitet, da das Land der Lieferanschrift vom Kunden abweicht."
                };
            }

            if (Input.BELE_L_STORNO)
            {
                yield return new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = "<strong>ACHTUNG!</strong> Der Auftrag wurde storniert und ist somit zur weiteren Bearbeitung gesperrt!"
                };
            }
            else if (Gesperrt && !Input.BELE_L_STORNO)
            {
                yield return new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Der Auftrag wurde bereits in einen Lieferschein übernommen und ist somit zur weiteren Bearbeitung gesperrt."
                };

            }
            else if (Input.BELE_L_ERLEDIGT && !Input.BELE_L_STORNO)
            {
                yield return new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Der Auftrag wurde bereits erledigt und ist somit zur weiteren Bearbeitung gesperrt."
                };
            }
            else if (GesperrtDurchZahlung && !Input.BELE_L_STORNO)
            {
                yield return new AlertBox
                {
                    AlertType = AlertType.Warning,
                    Message = "Der Auftrag wurde bereits bezahlt und ist somit zur weiteren Bearbeitung gesperrt."
                };
            }

        }
        private async Task UploadLieferscheinAsync(InputFileChangeEventArgs e)
        {
            string? extension = Path.GetExtension(e.File.Name);

            if (extension is not null && extension.Equals(".PDF", StringComparison.OrdinalIgnoreCase))
            {
                if (e.File.Size > 5000000)
                {
                    await jsRuntime.ShowToast(ToastType.error, $"Die Datei darf nicht größer als 5MB sein");
                    return;
                }
                byte[] buffer = new byte[e.File.Size];
                await using MemoryStream stream = new MemoryStream(buffer);
                await e.File.OpenReadStream(e.File.Size).CopyToAsync(stream);
                Input.NeutralerLieferschein = buffer;
            }
        }

        private async Task DownloadNeutralerLieferscheinAsync()
        {
            if (Input.NeutralerLieferschein.Length > 0)
            {
                await downloadService.DownloadFile($"NLS_{Auftragsnummer}.pdf", Input.NeutralerLieferschein, "application/pdf");
                await downloadService.ClearBuffers();
            }
        }

        private async Task AuftragErledigenAsync()
        {
            if (_editForm is null || _editForm.EditContext is null)
            {
                return;
            }

            if (_editForm.EditContext.Validate())
            {
                await SubmitAsync(true);
            }
            else
            {
                await jsRuntime.ShowToast(ToastType.error, "Der Auftrag konnte nicht auf erledigt gesetzt werden, da die Validierung fehlgeschlagen ist");
            }


        }

        public async Task OnTabChange(AufträgeTabs newActiveTab)
        {
            switch (newActiveTab)
            {
                case AufträgeTabs.Positionen:
                    if (Input.Positionen.Count is 0)
                    {
                        _showArtikelSucheModal = true;
                    }
                    break;
                case AufträgeTabs.Historie:
                    await LadeHistorieAsync();
                    break;
                default:
                    break;
            }
        }

        public override void AddHotKeys(HotKeysContext context)
        {
            context.Add(ModKeys.Alt, Keys.S, SubmitFormAsync, "Speichert den Auftrag", Exclude.Default);
            base.AddHotKeys(context);
        }
    }
}
