﻿using Microsoft.AspNetCore.Components;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;
using WK5.Core.Models.API;
using WK5.Core.PageModels.Tools.Regalfinder;

namespace WK5_Blazor.Pages.Tools
{
    public partial class RegalFinder
    {
        [Parameter] public bool ComponentView { get; set; } = false;

#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable
        [Parameter] public RegalFinderInput Input { get; set; } = new RegalFinderInput();
        private ArtikelFilter Filter { get; set; } = new ArtikelFilter();

        private List<ArtikelAssoziation> _assoziationen = new List<ArtikelAssoziation>();

        private ArtikelAssoziation? _selectedAssoziationsnummer;

        private bool showArtikelnummerSuchenModal;
        private bool _scanLagerplatz = true;

        private async Task ArtikeldatenÄndern()
        {
            RegalFinderResponse? result = Input.Modus switch
            {
                RegalFinderModus.LagerplatzÄndern => await regalfinderService.ChangeLagerplatzAsync(Input.Artikelnummer, Input.Lagerplatz),
                RegalFinderModus.EanÄndern => await regalfinderService.ChangeEanAsync(Input.Artikelnummer, Input.Ean),
                RegalFinderModus.HerstellernummerÄndern => await regalfinderService.ChangeHerstellernummerAsync(Input.Artikelnummer, Input.Herstellernummer),
                RegalFinderModus.AssoziationsnummerHinzufügen => await regalfinderService.AssoziationsnummerHinzufügenAsync(Input.Artikelnummer, Input.AlternativArtikelnummer),
                RegalFinderModus.LagerplatzSuchen => await regalfinderService.GetCurrentLagerplatzAsync(Input.Artikelnummer),
                RegalFinderModus.ArtikeletikettDrucken => await regalfinderService.DruckEtikettAsync(Input.Artikelnummer, Input.Menge),
                _ => null
            };

            if (result is not null)
            {
                await jsRuntime.ShowSweetAlert2("Erfolg", "Assoziationsnummer erfolgreich hinzugefügt", SweetAlertMessageType.success, 2000);
            }

            if (Input.Modus == RegalFinderModus.AssoziationsnummerHinzufügen)
            {
                Input.AlternativArtikelnummer = Input.AlternativArtikelnummer.ToUpper();
                ArtikelAssoziation assoziation = new ArtikelAssoziation(Input.Artikelnummer, Input.AlternativArtikelnummer);


                if (!_assoziationen.Contains(assoziation))
                {
                    _assoziationen.Add(assoziation);
                }
            }

            Input = new RegalFinderInput
            {
                Modus = Input.Modus,
                Artikelnummer = ComponentView ? Input.Artikelnummer : String.Empty
            };


            if (result is null)
            {
                await jsRuntime.ShowSweetAlert2(
                    title: "Oops",
                    message: $"Modus unbekannt. Bitte wenden Sie sich an die Programmierung.",
                    type: SweetAlertMessageType.error
                );
            }
            else
            {
                await jsRuntime.ShowSweetAlert2(
                    title: result.Success ? "Erfolg" : "Oops",
                    message: result.Message ?? String.Empty,
                    type: result.Success ? SweetAlertMessageType.success : SweetAlertMessageType.error,
                    timer: Input.Modus == RegalFinderModus.LagerplatzSuchen ? 0 : 2000
                );
            }


        }
        private string GetModusDescription(RegalFinderModus modus) => modus switch
        {
            RegalFinderModus.ArtikeletikettDrucken => "Etikett drucken",
            RegalFinderModus.AssoziationsnummerHinzufügen => "Assoziationsnummer hinzufügen",
            RegalFinderModus.EanÄndern => "EAN ändern",
            RegalFinderModus.HerstellernummerÄndern => "Herstellernummer ändern",
            RegalFinderModus.LagerplatzSuchen => "Lagerplatz suchen",
            RegalFinderModus.LagerplatzÄndern => "Lagerplatz ändern",
            _ => throw new ArgumentOutOfRangeException(nameof(modus))
        };

        private string GetConfirmButtonDescription(RegalFinderModus modus) => modus switch
        {
            RegalFinderModus.AssoziationsnummerHinzufügen => "Hinzufügen",
            RegalFinderModus.ArtikeletikettDrucken => "Etikett drucken",
            RegalFinderModus.LagerplatzSuchen => "Artikel suchen",
            _ => "Speichern"
        };
        private void OnArtikelSelected(Artikel artikel)
        {
            Input.Artikelnummer = artikel.Artikelnummer;
            showArtikelnummerSuchenModal = false;
        }

        private Task LagerplatzInputFocusInAsync()
        {
            Input.Lagerplatz = string.Empty;
            return Task.CompletedTask;
        }
        private async Task DeleteAssoziation()
        {
            if (_selectedAssoziationsnummer is not null)
            {
                await regalfinderService.DeleteAssoziationsnummerAsync(_selectedAssoziationsnummer.Assoziation);
                _assoziationen.Remove(_selectedAssoziationsnummer);
                _selectedAssoziationsnummer = null;
            }
        }

        protected override async Task OnInitializedAsync()
        {
            _assoziationen = await regalfinderService.GetAssoziationsnummernAsync().ToListAsync();
        }
    }
}
