﻿using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Server.ProtectedBrowserStorage;
using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;
using WK5.Core.PageModels.Stammdaten.Kunden;
using WK5_Blazor.CoreExtensions;

namespace WK5_Blazor.Pages.Stammdaten.Kunden
{
    public partial class KundenÜbersicht : IHasPageFilter, IAsyncDisposable
    {
        public const string LocalStorageFilterKey = "KundenÜbersicht.Filter";
        private int _seite = 1;
        private CancellationTokenSource _cts = new();
        private CancellationTokenSource _ctsAnsprechpartner = new();
        private bool _lädt = false;
        private string? _selectedKundennummer;
        private bool _lädtAnsprechpartner;
        private bool _showKundenkontakte = false;
        private bool _showAnsprechpartner = false;
        
        [Parameter] public int Seite { get => _seite; set => _seite = value <= 0 ? 1 : value; }
        public KundenÜbersichtFilter Filter { get; set; } = new KundenÜbersichtFilter();
        public int TotalItems { get; set; }
        public KundenBelegeFilter? FilterKundenBelege { get; set; }
        /// <summary>
        /// Eine Liste mit Kunden die in der Übersicht angezeigt wird
        /// </summary>
        public List<WK5.Core.Models.KundenÜbersicht> Übersicht { get; set; } = new List<WK5.Core.Models.KundenÜbersicht>();
        private List<AnsprechpartnerInput> AnsprechpartnerInputs { get; } = new List<AnsprechpartnerInput>();

        protected override async Task OnInitializedAsync()
        {
            await SetAlerts();
        }
        protected override async Task OnParametersSetAsync()
        {
            var localStorageResult = await LocalStorage.GetAsync<KundenÜbersichtFilter>(LocalStorageFilterKey);
            if (localStorageResult.Success)
            {
                if (localStorageResult.Value is not null)
                {
                    Filter = localStorageResult.Value;
                }
            }

            Filter.Seite = Seite;

            await LadeÜbersicht();
        }
        /// <summary>
        /// Lädt die Kunden in die Übersicht
        /// </summary>
        /// <returns>Gibt einen leeren Task zurück</returns>
        private async Task LadeÜbersicht(bool navigateToPage1 = false)
        {
            if (_lädt)
            {
                return;
            }

            _lädt = true;
            if (navigateToPage1)
            {
                Seite = 1;
            }
            Übersicht.Clear();
            _cts.Cancel();
            _cts.Dispose();
            _cts = new CancellationTokenSource();
            Filter.Seite = Seite;
            await LocalStorage.SetAsync(LocalStorageFilterKey, Filter);

            var token = _cts.Token;
            try
            {
                TotalItems = await kundenService.GetKundenÜbersichtAnzahlAsync(Filter, token);

                await foreach (var item in kundenService.GetKundenÜbersichtAsync(Filter, token))
                {
                    Übersicht.Add(item);
                    StateHasChanged();
                }
            }
            catch (Exception)
            {
                Filter = new KundenÜbersichtFilter();
                await LocalStorage.SetAsync(LocalStorageFilterKey, Filter);
                throw;
            }

            _lädt = false;
        }
        private Task ShowKundenKontakte(string kundennummer)
        {
            _selectedKundennummer = kundennummer;
            _showKundenkontakte = true;
            return Task.CompletedTask;
        }
        private async Task ShowAnsprechpartner(string kundennummer)
        {
            _ctsAnsprechpartner.Cancel();
            _ctsAnsprechpartner = new CancellationTokenSource();
            _selectedKundennummer = kundennummer;
            _showAnsprechpartner = true;
            _lädtAnsprechpartner = true;

            var token = _ctsAnsprechpartner.Token;
            await Task.Run(async () =>
            {
                await foreach (var partner in Ansprechpartner.GetAnsprechpartnerAsync(kundennummer, token))
                {
                    AnsprechpartnerInputs.Add(partner.ToInput());
                    await InvokeAsync(StateHasChanged);
                }
            }, token);


            _lädtAnsprechpartner = false;
        }
        private async Task UpdateWichtigkeitAsync(string kundennummer) => await kundenService.UpdateWichtigkeitAsync(kundennummer);
        private Task ShowKundenBelege(WK5.Core.Models.KundenÜbersicht kunde)
        {
            FilterKundenBelege = new KundenBelegeFilter()
            {
                Suchbegriff = kunde.KUNU_A_NR
            };

            return Task.CompletedTask;
        }
        public override ValueTask DisposeAsync()
        {
            _cts.Cancel();
            _cts.Dispose();
            return base.DisposeAsync();
        }
    }
}
