﻿using ecoDMS.Library;
using MbeSoap;
using Microsoft.AspNetCore.Components;
using Microsoft.AspNetCore.Components.Forms;
using Serilog;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;
using WK5.Core.Models.Lager;
using WK5.Core.Models.Tools.Lagerregal;
using WK5.Core.Models.Versand;
using WK5.Core.Validators.Lager;
using KarleyLibrary.Erweiterungen;
using Toolbelt.Blazor.HotKeys;

namespace WK5_Blazor.Pages.Lager.Zugänge
{
    public partial class RechnungZugang : IEditPage
    {
#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable
        [Parameter] public int Zugangsnummer { get; set; }
        [Parameter] public int RmaNummer { get; set; }
        [Parameter] public bool RedirectToNewLieferschein { get; set; }

        #region Filter 
        private LieferantenFilter FilterLieferanten { get; set; } = new LieferantenFilter();
        private BestellungFilter FilterBestellungen { get; set; } = new BestellungFilter
        {
            Option = BestellungFilterOption.NurOffeneBestellungen
        };
        private ArtikelFilter FilterArtikel { get; set; } = new ArtikelFilter();
        #endregion
        public RechnungBuchenInput Input { get; set; } = new RechnungBuchenInput(false);
        public RechnungBuchenInput StartCopy { get; set; } = new RechnungBuchenInput(false);
        public string ActivePageName => Modus switch
        {
            EditMode.Anlage => "Rechnung zubuchen",
            EditMode.Bearbeiten => $"Zugang {Zugangsnummer} bearbeiten",
            _ => "Unbekannt"
        };
        public string FinalBreadcrumbItemName => Modus switch
        {
            EditMode.Anlage => "Rechnung zubuchen",
            EditMode.Bearbeiten => $"{Zugangsnummer}",
            _ => "Unbekannt"
        };
        public PackAuftrag? DirektlieferungAuftrag { get; set; }
        public Lieferant? Lieferant { get; set; }
        public EditMode Modus { get; set; }
        public DirektlieferungFortführenInput DirektlieferungInput { get; set; } = new DirektlieferungFortführenInput();
        private PositionBuchenInput? DeleteSelectedPositionsInput { get; set; }
        private Bestellung? SelectedBestellung { get; set; }

        private bool _showBestellungSucheModal;
        private bool _showArtikelSucheModal;
        private bool _showLieferantenSucheModal;
        private bool _showDirektlieferungFortführenModal;
        private bool _isSaving;
        private Rma? _rma;
        private bool _hideBestellungÜbernehmenButton;
        private bool _hideAddArtikelButton;
        private DirektlieferungFortführenInputValidator _direktlieferungValidator = new DirektlieferungFortführenInputValidator(new DirektlieferungPaketInputValidator());
        protected override async Task OnParametersSetAsync()
        {
            if (Zugangsnummer > 0)
            {
                await LadeEditModeAsync();
            }

            if (navigationManager.Uri.ToString().Contains("InternerZugang"))
            {
                if (!Mitarbeiter.IsAdmin)
                {
                    await AddAlertAsync(new AlertBox("Diese Seite ist nur für Administratoren", AlertType.Warning));
                    navigationManager.NavigateTo("/Zugang");
                    return;
                }
                else
                {
                    Input.InternerZugang = true;
                    FilterLieferanten.NurEigenlieferantenAnzeigen = true;
                }
            }
            else
            {
                FilterLieferanten.NurNichtEigenlieferantenAnzeigen = true;
            }

            StartCopy = Input.DeepCopy();


            if (Modus is EditMode.Anlage && RmaNummer > 0)
            {
                using FbController2 fbController = new FbController2();

                _rma = await rmaService.GetAsync(RmaNummer, fbController);
                if (_rma is not null)
                {
                    Input.RmaNummerVerknüpfung = RmaNummer;
                    Input.Rechnungsnummer = $"RMA-{RmaNummer}";
                    Input.Rechnungsdatum = _rma.Anlagedatum;


                    await SelectLieferant(await Lieferant.GetLieferantAsync(_rma.LieferantId));

                    var artikel = await artikelService.GetAsync(_rma.Artikelnummer, fbController);
                    if (artikel is not null)
                    {
                        Belegcharge belegcharge = await Belegcharge.GetBelegchargeAsync(_rma.BelegChargenId) ?? throw new NullReferenceException(nameof(belegcharge));

                        Charge charge = await Charge.GetChargeAsync(belegcharge.BCHA_N_CHARGE, fbController) ?? throw new NullReferenceException(nameof(charge));

                        var newPos = await AddArtikel(artikel);
                        newPos.Menge = _rma.Menge;
                        newPos.Preis = charge.CHAR_N_EKPREIS;
                        if (newPos.SeriennummerBenötigt)
                        {
                            newPos.Seriennummern[0].Seriennummer = _rma.Seriennummer;
                        }

                    }

                    Input.GesamtbetragNetto = Input.Positionen.Sum(x => x.Preis * x.Menge);
                }
            }

            if (Lieferant is null)
            {
                _showLieferantenSucheModal = true;
            }


        }
        public async Task LadeEditModeAsync()
        {
            var zugang = await Zugang.GetZugangAsync(Zugangsnummer);
            if (zugang is not null)
            {
                await SelectLieferant(await Lieferant.GetLieferantAsync(zugang.ZUGA_N_LIEF));
                Modus = EditMode.Bearbeiten;
                Input = await zugang.ToInputAsync();


            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Zugang {Zugangsnummer} konnte nicht gefunden werden"
                });

                navigationManager.NavigateTo("/Zugang");
            }

            if (!_isDisposed)
            {
                await CheckActivePageAsync();
                await SetAlerts();
            }
        }
        private async Task SubmitAsync()
        {
            if ((Input.Gesperrt && !(Mitarbeiter.IsAdmin || Mitarbeiter.IsBuchhaltung)) || SperreDurchAnderenUser || _isSaving || !Input.HasBeenModified(StartCopy))
            {
                return;
            }

            if (Modus is EditMode.Anlage && IstDirektlieferungFortführenMöglich())
            {
                _showDirektlieferungFortführenModal = true;
                return;
            }

            await SaveAsync();
        }
        private async Task SaveAsync(bool direktlieferungFortführen = false)
        {

            if (Lieferant is null || (Input.Gesperrt && !Mitarbeiter.IsBuchhaltung))
            {
                return;
            }

            bool umleitenInBestellung = false;

            _isSaving = true;
            StateHasChanged(); // Wird für den Shortcut benötigt
            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await fbController.StartTransactionAsync();
            try
            {
                if (Modus is EditMode.Anlage)
                {
                    int zugangsnummer = await lagerService.CreateZugangAsync(fbController, Mitarbeiter.PersonalNummer, Input, Lieferant);
                    Input.Zugangsnummer = zugangsnummer;

                    if (_rma is not null)
                    {
                        await rmaService.SetRmaZugangAsync(RmaNummer, zugangsnummer, fbController);
                        await lagerService.CloseRmaZugangAsync(RmaNummer, fbController);
                        if (RedirectToNewLieferschein)
                        {
                            await rmaService.SetZurückVonLieferant(RmaNummer, fbController);
                        }
                    }

                    if (Input.InternerZugang)
                    {
                        await lagerService.ZugangAbschließen(zugangsnummer,fbController);
                    }
                }

                if (!String.IsNullOrWhiteSpace(Input.Rechnungsdatei) && File.Exists(Input.Rechnungsdatei))
                {
                    try
                    {
                        EcoDmsController ecoController = await EcoDmsController.CreateAsync(new Uri(GlobalConfig.ECODMS_BASIS_URL), $"{Mitarbeiter.Username}:{Mitarbeiter.Passwort}", Log.Logger);
                        bool angemeldet = await ecoController.AnmeldenAsync();
                        if (angemeldet)
                        {
                            // Wir prüfen erst, ob das Dokument bereits in ecoDMS archiviert worden ist
                            List<EcoSearchFilter> suchbedingungen = new List<EcoSearchFilter>
                            {
                                new EcoSearchFilter
                                {
                                    ClassifyAttribut = ecoController.ClassifyAttributes.GetKeyByValue("Belegnummer")!,
                                    SearchOperator = SuchOperator.Gleich,
                                    SearchValue = Input.Rechnungsnummer
                                },
                                new EcoSearchFilter
                                {
                                    ClassifyAttribut = "docart",
                                    SearchOperator = SuchOperator.Gleich,
                                    SearchValue = "15"
                                }
                            };

                            List<EcoDMSdocInfoCL> result = await ecoController.EcoSucheDoksAsync(suchbedingungen);
                            EcoDMSdocInfoCL? ecoDokument = result.FirstOrDefault();
                            if (ecoDokument is null)
                            {
                                // Rechnung in ecoDMS hochladen
                                // TODO: (BIG) Felder und Eigenschaften per Einstellungen Konfigurierbar
                                Dictionary<string, string> klassifizierung = new Dictionary<string, string>
                                {
                                    {"docart", "15" },
                                    {"folder", "3" },
                                    {"mainfolder", "3" },
                                    {"status", "1" },
                                    {ecoController.ClassifyAttributes.GetKeyByValue("Kontonummer")!, Lieferant.LIEF_N_NR.ToString() },
                                    {"cdate", Input.Rechnungsdatum.ToString("yyyy-MM-dd") },
                                    {ecoController.ClassifyAttributes.GetKeyByValue("Steuerberater")!, "2" },
                                    {"bemerkung", Lieferant.LIEF_A_NAME1 },
                                    {ecoController.ClassifyAttributes.GetKeyByValue("ToDo")!, Program.AppDaten.Zahlungsbedingungen.IstLastschrift(Lieferant.LIEF_N_ZABD) ? "Datev verbuchen" : "Datev & zu zahlen" },
                                    {ecoController.ClassifyAttributes.GetKeyByValue("Belegnummer")!, Input.Rechnungsnummer },
                                };

                                List<string> editRoles = new List<string>
                                {
                                    "r_LDAP_Buchhaltung", "r_ecodms", "ecoSIMSADMIN", "LDAP_r_ley"
                                };

                                List<string> readRoles = new List<string>
                                {
                                    "ecoSIMSUSER"
                                };

                                int docId = await ecoController.UploadFileAsync(Input.Rechnungsdatei, klassifizierung, readRoles, editRoles);

                                if (docId > 0)
                                {
                                    await lagerService.SetEcoDmsDocIdAsync(fbController, Input.Zugangsnummer, docId);
                                }
                            }
                            else
                            {
                                await AddAlertAsync(new AlertBox
                                {
                                    AlertType = AlertType.Warning,
                                    Message = $"Die Rechnung {Input.Rechnungsnummer} wurde bereits in ecoDMS gefunden. Die Rechnung wurde daher nicht erneut archiviert."
                                });
                            }
                            await ecoController.AbmeldenAsync();
                        }
                    }
                    catch (Exception ex)
                    {
                        await AddAlertAsync(new AlertBox
                        {
                            AlertType = AlertType.Danger,
                            Message = $"Die Rechnung konnte nicht in ecoDMS archiviert werden. Bitte Die Rechnung manuell archivieren.{Environment.NewLine}Fehler: {ex}"
                        });
                    }

                }


                await lagerService.UpdateZugangAsync(fbController, Input, Lieferant);

                Rechnung? rechnung = null;
                if (DirektlieferungAuftrag is not null && direktlieferungFortführen)
                {
                    DirektlieferungAuftrag = await PackAuftrag.GetPackAuftragAsync(DirektlieferungAuftrag.Belegnummer, fbController) ?? throw new ArgumentNullException(nameof(DirektlieferungAuftrag));

                    await auftragsfreigabeService.AuftragFreigebenAysnc(DirektlieferungAuftrag.Belegnummer, DateTime.Now, fbController);
                    foreach (var pos in DirektlieferungAuftrag.GetEndPositionen())
                    {
                        if (pos.PositionBenötigtSeriennummer)
                        {
                            var sucheSeriennummern = Input.Positionen.Where(x => x.Artikelnummer.Equals(pos.Artikelnummer)).FirstOrDefault();
                            if (sucheSeriennummern is null)
                            {
                                throw new ArgumentNullException();
                            }

                            foreach (var sn in sucheSeriennummern.Seriennummern)
                            {
                                var response = await buchenService.BuchenAsync(pos, 1, pos.Artikelnummer, sn.Seriennummer, fbController);
                            }

                        }
                        else
                        {
                            var response = await buchenService.BuchenAsync(pos, pos.Menge, pos.Artikelnummer, null, fbController);
                        }
                    }

                    var lieferschein = await DirektlieferungAuftrag.ÜbernahmeInLieferscheinAsync(fbController, Program.AppDaten.Optionen, belegService);
                    await lieferschein.ChargenAsync(fbController, Mitarbeiter);
                    rechnung = await lieferschein.ÜbernahmeInRechnungAsync(fbController, Program.AppDaten.Optionen, belegService);
                    await lagerregalService.InsertGepacktAsync(DirektlieferungAuftrag, lieferschein, rechnung, fbController);

                }

                await fbController.CommitChangesAsync();

                if (rechnung is not null)
                {
                    VersandOption option = new VersandOption(DirektlieferungInput.Versender, CourierArt.MANUELL);

                    List<string> trackingnummern = DirektlieferungInput.Pakete.Select(x => x.Trackingnummer).ToList();

                    trackingnummern.RemoveAll(x => x.Equals("OHNE"));


                    VersandMailer versandMailer = await VersandMailer.CreateAsync(rechnung, trackingnummern, option, Mitarbeiter.PersonalNummer, fbController);


                    var result = await versandMailer.EmailSendenAsync();

                    if (result.Where(x => !x.Success).Any())
                    {
                        await jsRuntime.ShowToast(ToastType.error, $"Fehler beim senden der Email. Fehler wurde an die info@karley.eu gesendet!", 5000);
                    }

                    RecipientType emp = await rechnung.ToRecipientTypeAsync(fbController);
                    foreach (var paket in DirektlieferungInput.Pakete)
                    {
                        await versandService.PaketInW4EintragenAsync(rechnung, paket.Trackingnummer, emp, paket.Gewicht);
                    }

                    //Scholz - 23.09.2020: Speichert in der Datenbank welche Belege Manuell versendet wurden, ist für uns übersichtlicher
                    await ManuellVersand.CreateAsync(rechnung.Belegtyp, rechnung.Belegnummer, true, false, Mitarbeiter.PersonalNummer);

                    await jsRuntime.ShowToast(ToastType.success, $"Sendung wurde erfolgreich eingetragen.", 3000);
                }


                if (Modus is EditMode.Anlage && Input.Bestellnummer > 0 && !RedirectToNewLieferschein)
                {
                    Bestellung? bestellung = await Bestellung.GetBestellungAsync(Input.Bestellnummer, fbController);
                    if (bestellung is not null)
                    {
                        // Wir prüfen nach, ob die Bestellung noch nicht erledigt worden ist.
                        // Wenn ja, soll der Mitarbeiter in die Bestellung umgeleitet werden, sodass er einen neuen Liefertermin hinterlegen kann
                        if (!bestellung.BEST_L_ERLEDIGT)
                        {
                            await AddAlertAsync(new AlertBox
                            {
                                AlertType = AlertType.Warning,
                                Message = "Bitte geben Sie für die Bestellung einen neuen Liefertermin an"
                            });

                            umleitenInBestellung = true;
                        }
                    }
                }

                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Success,
                    Message = Modus is EditMode.Anlage
                        ? $"Zugang {Input.Zugangsnummer} erfolgreich gebucht."
                        : $"Zugang {Input.Zugangsnummer} erfolgreich gespeichert."
                });


            }
            catch (Exception ex)
            {
                await fbController.RollbackChangesAsync();
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Fehler beim Speichern des Zugangs: {ex}"
                });
            }
            _isSaving = false;

            if (RmaNummer > 0 && RedirectToNewLieferschein)
            {
                navigationManager.NavigateTo($"/Lieferscheine/Neu/{RmaNummer}/2");
            }
            else if (umleitenInBestellung)
            {
                navigationManager.NavigateTo($"/Bestellungen/{Input.Bestellnummer}");
            }
            else
            {
                navigationManager.NavigateTo("/Zugang");
            }

        }
        private async Task BestellungÜbernehmen(Bestellung bestellung)
        {
            _hideAddArtikelButton = true;
            _hideBestellungÜbernehmenButton = true;

            using FbController2 fbController = new FbController2();
            if (bestellung.WK5_BELE_L_DIREKTLIEFERUNG)
            {
                PackAuftrag? auftrag = await PackAuftrag.GetPackAuftragAsync(bestellung.BEST_N_AB_NR, fbController);

                if (auftrag is not null)
                {
                    // 28.01.2021 Wie mit LEY und SOCKE besprochen, ist eine Automatischer Erstellung des Lieferscheins und Rechnung 
                    // nur dann möglich, wenn Alle Positionen mit Lagerführung 1:1 mit denen der Bestellung übereinstimmen
                    // Teillieferungen werden von dem System nicht erfasst und werden manuell vom Mitarbeiter durchgeführt
                    bool direktLieferungFortführenMöglich = true;
                    foreach (var pos in auftrag.GetEndPositionen())
                    {

                        // Artikel ohne Lagerführung sind egal und werden einfach übernommen
                        if (!pos.ARTI_L_LAGERFUEHR)
                        {
                            continue;
                        }

                        // Wenn bereits ein Teil ausgliefert worden ist, ist eine Erstellung des Lieferscheins und der Rechnung nicht mehr möglich
                        if (pos.BPOS_N_MENGEGELIEF > 0)
                        {
                            direktLieferungFortführenMöglich = false;
                            break;
                        }

                        var sucheArtikelInBestellung = bestellung.Positionen.Where(x => x.BEPO_A_ARTIKELNR.Equals(pos.Artikelnummer)).FirstOrDefault();

                        // Artikel aus Auftrag ist nicht in Bestellung vorhanden, also eine Erstellung des Lieferscheins und der Rechnung ist nicht möglich
                        if (sucheArtikelInBestellung is null)
                        {
                            direktLieferungFortführenMöglich = false;
                            break;
                        }
                        else
                        {
                            if (sucheArtikelInBestellung.BEPO_N_MENGE != pos.Menge)
                            {
                                direktLieferungFortführenMöglich = false;
                                break;
                            }
                        }
                    }

                    if (direktLieferungFortführenMöglich)
                    {
                        DirektlieferungAuftrag = auftrag;
                    }
                }
            }

            foreach (var pos in bestellung.Positionen)
            {
                if (!pos.BEPO_L_ERLEDIGT)
                {
                    Artikel? artikel = await artikelService.GetAsync(pos.BEPO_A_ARTIKELNR, fbController);

                    if (artikel is null)
                    {
                        continue;
                    }


                    decimal offen = pos.BEPO_N_MENGE - pos.BEPO_N_LIEFERMENGE;
                    decimal bezahlterBedarf = await artikelService.GetBedarfBezahlt(artikel.Artikelnummer, Program.AppDaten.Zahlungsbedingungen);
                    Input.Positionen.Add(new PositionBuchenInput(artikel.ARTI_L_SN, offen)
                    {
                        Artikelnummer = pos.BEPO_A_ARTIKELNR,
                        Bestellnummer = pos.BEPO_N_BESTNR,
                        BestellungPosId = pos.BEPO_N_POS,
                        Preis = pos.PreisMitRabatt,
                        Menge = offen,
                        ArtikelnummerLieferant = pos.BEPO_A_BESTELLNUMMER ?? String.Empty,
                        Bezeichnung = pos.GetBezeichnung(),
                        MwstSatz = pos.BEPO_N_MWSTPROZ,
                        BezahlterBedarf = bezahlterBedarf,
                        Auftragsnummer = bestellung.BEST_N_AB_NR
                    });
                }
            }
            Input.Zusatzkosten += bestellung.Zusatzkosten;
            FilterBestellungen.ZeigeKeineDirektlieferungen = true;
            Input.BestellungNotiz = new ZugangBestellnotizInput(bestellung.Bestellnummer, bestellung.BEST_B_NOTIZ);
            Input.Bestellnummer = bestellung.Bestellnummer;
            _showBestellungSucheModal = false;
        }
        private async Task<PositionBuchenInput> AddArtikel(Artikel artikel)
        {
            _hideBestellungÜbernehmenButton = true;
            _showArtikelSucheModal = false;

            decimal bezahlterBedarf = await artikelService.GetBedarfBezahlt(artikel.Artikelnummer, Program.AppDaten.Zahlungsbedingungen);
            var newPos = new PositionBuchenInput(artikel.ARTI_L_SN, artikel.Bestellt)
            {
                Artikelnummer = artikel.Artikelnummer,
                Menge = artikel.Bestellt,
                Bezeichnung = artikel.Bezeichnung1,
                MwstSatz = artikel.MWST_N_PROZENT,
                BezahlterBedarf = bezahlterBedarf,
                Preis = artikel.DurchschnittsEinkaufspreis
            };
            Input.Positionen.Add(newPos);

            FilterBestellungen.ZeigeKeineDirektlieferungen = true;
            return newPos;
        }
        private Task SelectBestellung(Bestellung bestellung)
        {
            SelectedBestellung = bestellung;
            return Task.CompletedTask;
        }
        private Task SelectLieferant(Lieferant? lieferant)
        {
            Lieferant = lieferant;

            if (Lieferant is not null)
            {
                FilterBestellungen.LieferantId = Lieferant.LIEF_N_NR;
                FilterArtikel.LieferantenId = Lieferant.LIEF_N_NR;
            }

            _showLieferantenSucheModal = false;
            return Task.CompletedTask;
        }
        private Task LieferantenModalClosed()
        {
            _showLieferantenSucheModal = false;
            if (Lieferant is null)
            {
                navigationManager.NavigateTo("/Zugang");
            }
            return Task.CompletedTask;
        }
        private bool IstDirektlieferungFortführenMöglich()
        {
            if (DirektlieferungAuftrag is null)
            {
                return false;
            }

            if (Input.Positionen.Count != DirektlieferungAuftrag.Positionen.Where(x => x.ARTI_L_LAGERFUEHR).Count())
            {
                return false;
            }

            foreach (var pos in Input.Positionen)
            {
                var sucheArtikelInAuftrag = DirektlieferungAuftrag.Positionen.Where(x => x.Artikelnummer.Equals(pos.Artikelnummer)).FirstOrDefault();
                if (sucheArtikelInAuftrag is null)
                {
                    return false;
                }
                else if (sucheArtikelInAuftrag.Menge != pos.Menge)
                {
                    return false;
                }
            }

            return true;
        }
        private async Task OnInputFileChange(InputFileChangeEventArgs e)
        {
            if (!Path.GetExtension(e.File.Name).Equals(".pdf", StringComparison.OrdinalIgnoreCase))
            {
                await jsRuntime.ShowToast(ToastType.error, "Upload fehlgeschlagen. Bitte laden Sie eine PDF-Datei hoch.");
                return;
            }

            string path = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Uploads");
            string filename = Path.Combine(path, e.File.Name);

            if (!Directory.Exists(path))
            {
                Directory.CreateDirectory(path);
            }

            await using FileStream fs = new(filename, FileMode.Create);
            await e.File.OpenReadStream(e.File.Size).CopyToAsync(fs);



            Input.Rechnungsdatei = filename;
        }
        private decimal GetBrutto()
        {
            if (Modus is EditMode.Anlage)
            {
                if (Lieferant is not null && Lieferant.LIEF_L_MWST)
                {
                    //return Input.GesamtBrutto(Program.AppDaten.VersandMehrwertsteuer);
                    return Input.GesamtBrutto(Lieferant.LIEF_L_MWST, Program.AppDaten.VersandMehrwertsteuer);
                }
                else
                {
                    //return Input.GesamtBrutto(null);
                    return Input.GesamtBrutto(false, null);
                }
            }
            else
            {
                if (Input.MwstBeiBearbeitung)
                {
                    //return Input.GesamtBrutto(Program.AppDaten.VersandMehrwertsteuer);
                    return Input.GesamtBrutto(Input.MwstBeiBearbeitung, Program.AppDaten.VersandMehrwertsteuer);
                }
                else
                {
                    //return Input.GesamtBrutto(null);
                    return Input.GesamtBrutto(false, Program.AppDaten.VersandMehrwertsteuer);
                }
            }
        }
        public override async Task CheckActivePageAsync()
        {
            var (success, page) = ActivePages.AddActivePage(new ActivePage(PageType.Zugang, Zugangsnummer.ToString(), Mitarbeiter.PersonalNummer));

            if (success)
            {
                Page = page;
            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Der Zugang wird zurzeit durch {Program.AppDaten.GetMitarbeiterName(page.UserId)} bearbeitet."
                });

                SperreDurchAnderenUser = true;
            }
        }
        public override void AddHotKeys(HotKeysContext context)
        {
            context
                .Add(ModKeys.Alt, Keys.S, async () =>
                 {
                     await SubmitAsync();
                     StateHasChanged();
                 }
                 , "Speichert den Zugang", Exclude.Default)
                .Add(ModKeys.Alt, Keys.Insert, () => { _showArtikelSucheModal = true; StateHasChanged(); }, "Fügt dem Zugang einen neuen Artikel hinzu", Exclude.Default)
                .Add(ModKeys.Alt, Keys.B, () => { _showBestellungSucheModal = true; StateHasChanged(); }, "Fügt dem Zugang einen neuen Artikel hinzu", Exclude.Default)
                ;
        }

        #region IEditPage rest
        public string GetNavListClass(bool active) => throw new NotImplementedException();
        public string GetSubNavListClass(bool active) => throw new NotImplementedException();
        public string GetTabClass(bool active) => throw new NotImplementedException();
        #endregion
    }
}
