﻿using Microsoft.AspNetCore.Components;
using System;
using System.Threading.Tasks;
using Toolbelt.Blazor.HotKeys;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;
using WK5.Core.Models.Lager;

namespace WK5_Blazor.Pages.Lager.Abgänge
{
    public partial class AbgangBuchen : IEditPage
    {
#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable
        public ArtikelFilter ArtikelFilter { get; set; } = new ArtikelFilter
        {
            NurArtikelMitLagerführung = true,
            ZeigeInaktive = false,
            NurArtikelMitBestand = true
        };

        public ChargenFilter ChargenFilter { get; set; } = new ChargenFilter()
        {
            NurOffene = true
        };

        public SeriennummerFilter SeriennummerFilter { get; set; } = new SeriennummerFilter()
        {
            NurNichtAusgelieferte = true
        };

        [Parameter] public int Abgangsnummer { get; set; }
        public string ActivePageName => Modus switch
        {
            EditMode.Anlage => "Abgang buchen",
            EditMode.Bearbeiten => $"Abgang {Abgangsnummer}",
            _ => "Unbekannt"
        };
        public string FinalBreadcrumbItemName => Modus switch
        {
            EditMode.Anlage => "Neuer Abgang",
            EditMode.Bearbeiten => $"Abgang {Abgangsnummer}",
            _ => "Unbekannt"
        };
        public EditMode Modus { get; set; }

        public AbgangBuchenInput Input { get; set; } = new AbgangBuchenInput();

        private Task OnArtikelnummerModalClosed()
        {
            if (String.IsNullOrWhiteSpace(Input.Artikelnummer))
            {
                navigationManager.NavigateTo("/Abgang");
            }
            return Task.CompletedTask;
        }

        private Task OnArtikelSelected(Artikel artikel)
        {
            Input.Artikelnummer = artikel.Artikelnummer;
            Input.BenötigtSeriennummer = artikel.ARTI_L_SN;
            ChargenFilter.Artikelnummer = artikel.Artikelnummer;
            SeriennummerFilter.Artikelnummer = artikel.Artikelnummer;
            return Task.CompletedTask;
        }

        private Task OnChargeSelected(Charge charge)
        {
            Input.Charge = charge.CHAR_N_NR;
            Input.MaxMenge = charge.CHAR_N_MENGEOFFEN;
            return Task.CompletedTask;
        }

        private Task OnChargenModalClosed()
        {
            if (Input.Charge is 0)
            {
                navigationManager.NavigateTo("/Abgang");
            }

            return Task.CompletedTask;
        }

        private Task OnSeriennummerSelected(Seriennummer seriennummer)
        {
            Input.MaxMenge = 1;
            Input.Menge = 1;
            Input.Seriennummer = seriennummer.Nummer;
            Input.Charge = seriennummer.Charge;
            return Task.CompletedTask;
        }

        private Task OnSeriennummerModalClosed()
        {
            if (Input.Charge is 0 || String.IsNullOrWhiteSpace(Input.Seriennummer))
            {
                navigationManager.NavigateTo("/Abgang");
            }

            return Task.CompletedTask;
        }

        protected override async Task OnParametersSetAsync()
        {
            if (Abgangsnummer > 0)
            {
                await LadeEditModeAsync();

            }
        }

        public async Task LadeEditModeAsync()
        {
            using FbController2 fbController = new FbController2();
            Abgang? abgang = await Abgang.GetAbgangAsync(Abgangsnummer, fbController);

            if (abgang is not null)
            {
                Input.Artikelnummer = abgang.ARAB_A_ARTIKEL;
                Input.Charge = abgang.ARAB_N_CHARGE;
                Input.Datum = abgang.ARAB_D_DATUM;
                Input.Kostenstelle = abgang.ARAB_N_KOSTENSTELL;
                Input.Menge = abgang.ARAB_N_MENGE;
                Input.Notiz = abgang.ARAB_A_NOTIZ ?? String.Empty;
                Modus = EditMode.Bearbeiten;
            }
            else
            {
                await AddAlertAsync(new AlertBox
                {
                    AlertType = AlertType.Danger,
                    Message = $"Abgang {Abgangsnummer} konnte nicht gefunden werden"
                });

                navigationManager.NavigateTo("/Abgang");
            }
        }

        private async Task SaveAsync()
        {
            if (Modus is EditMode.Bearbeiten)
            {
                return;
            }

            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);

            await fbController.StartTransactionAsync();

            try
            {
                int abgangsnummer = await lagerService.CreateAbgangAsync(Input, Mitarbeiter.PersonalNummer, fbController);
                await fbController.CommitChangesAsync();
                await AddAlertAsync(new AlertBox { AlertType = AlertType.Success, Message = $"Abgang {abgangsnummer} erfolgreich hinzugefügt." });
            }
            catch (Exception)
            {
                await fbController.RollbackChangesAsync();
                throw;
            }

            navigationManager.NavigateTo("/Abgang");
        }


        #region Interface IEditPage Rest
        public string GetNavListClass(bool active)
        {
            throw new NotImplementedException();
        }

        public string GetSubNavListClass(bool active)
        {
            throw new NotImplementedException();
        }

        public string GetTabClass(bool active)
        {
            throw new NotImplementedException();
        }


        #endregion

        public override void AddHotKeys(HotKeysContext context)
        {
            context.Add(ModKeys.Alt, Keys.S, SaveAsync, "Speichert den Abgang", Exclude.Default);
        }
    }
}
