﻿using MbeSoap;
using Microsoft.AspNetCore.Components;
using System;
using System.Linq;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;
using WK5.Core.PageModels.Einkauf.Bestellungen;

namespace WK5_Blazor.Pages.Bestellungen
{
    public partial class BestellungPrüfen : IPipeline<BestellungPrüfenInput>
    {

#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable
        private Kunde? _kunde;
        private bool _showSonderartikelConfirmModal;
        private RecipientType? _lieferanschrift;
        public BestellungFilter Filter { get; set; } = new BestellungFilter
        {
            Option = BestellungFilterOption.OffenUndNichtBestätigt
        };

        public Bestellung? Bestellung { get; set; }

        private Bestellung? SelectedBestellung { get; set; }


        public bool AllesGeprüft => Bestellung is not null && Bestellung.Positionen.Where(x => x.Geprüft).Count() == Bestellung.Positionen.Count;

        private BestellungPrüfenInput Input { get; set; } = new BestellungPrüfenInput();
        public bool PrüfenGesperrt { get; set; }
        public Pipeline<BestellungPrüfenInput> Pipeline { get; set; } = new Pipeline<BestellungPrüfenInput>();

        protected override async Task OnInitializedAsync()
        {
            await SetupPipelineAsync();
        }
        private Task BestellungClicked(Bestellung bestellung)
        {
            SelectedBestellung = bestellung;
            StateHasChanged();
            return Task.CompletedTask;
        }
        private async Task SelectBestellung(Bestellung bestellung)
        {
            Bestellung = bestellung;
            PrüfenGesperrt = bestellung.BEST_L_ERLEDIGT;
            Input = new BestellungPrüfenInput
            {
                Bestellnummer = Bestellung.Bestellnummer,
                Liefertermin = Bestellung.BEST_D_LIEFTERMIN,
                Frachtkosten = bestellung.Versandkosten,
                Mindermengenzuschlag = bestellung.Mindermengenzuschlag,
                Verpackungskosten = bestellung.Verpackungskosten,
                Versicherungskosten = bestellung.Versicherungskosten
            };

            if (!String.IsNullOrWhiteSpace(bestellung.BEST_A_LIEFERKUNDE))
            {
                _kunde = await Kunde.GetKundeAsync(bestellung.BEST_A_LIEFERKUNDE);
                if (bestellung.BEST_N_AB_NR > 0)
                {
                    using FbController2 fbController = new FbController2();
                    Auftrag? auftrag = await Auftrag.GetAuftragAsync(bestellung.BEST_N_AB_NR, fbController);

                    if (auftrag is not null)
                    {
                        _lieferanschrift = await auftrag.ToRecipientTypeAsync(fbController);
                    }
                }

            }


        }

        private Task ModalClosed()
        {
            if (Bestellung is null)
            {
                navigationManager.NavigateTo("Bestellungen");
            }
            return Task.CompletedTask;
        }

        private async Task SubmitAsync()
        {
            Pipeline.Reset();
            await Pipeline.RunUntilFailureAsync(Input);
        }

        private async Task SetGeprüftAsync()
        {

            using FbController2 fbController = new FbController2(Mitarbeiter.PersonalNummer);
            await bestellService.SetGeprüftAsync(Input, fbController);

            await AddAlertAsync(new AlertBox
            {
                AlertType = AlertType.Success,
                Message = $"Bestellung {Input.Bestellnummer} wurde erfolgreich geprüft",
                DecayTime = 10
            });

            navigationManager.NavigateTo("/Bestellungen");
        }

        public async Task SetupPipelineAsync()
        {
            await Pipeline.HardReset();

            Pipeline.Add((input) =>
            {
                if (Bestellung is not null && Bestellung.Positionen.Any(x => x.BEPO_A_ARTIKELNR.StartsWith("SO-")))
                {
                    _showSonderartikelConfirmModal = true;
                    return false;
                }

                return true;
            });

            Pipeline.Add(async (input) =>
            {
                await SetGeprüftAsync();
                return true;
            });
        }
    }
}

