﻿using Microsoft.AspNetCore.Components;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis;
using WK5.Core.Basis.Erweiterungen;
using WK5.Core.Basis.Filter;
using WK5.Core.Drucken.Briefe;
using WK5.Core.Models;

namespace WK5_Blazor.Components
{
    public partial class LieferantenKontakte : IAsyncDisposable
    {
        [Parameter] public int LieferantenId { get; set; }
        [Parameter] public LieferantenkontaktFilter Filter { get; set; } = new();
        [Parameter] public int Belegnummer { get; set; }
        [Parameter] public string Belegtyp { get; set; } = string.Empty;
        public List<LieferantenKontakt> Kontakte { get; set; } = new();
        public int TotalItems { get; set; }

        private LieferantenKontakt? _selectedLieferantenKontakt;

        private LieferantenKontakt? _downloadKontakt;

        private CancellationTokenSource _cts = new CancellationTokenSource();
        private bool _lädt;
        private async Task LadeAsync(int seite = 1)
        {
            _cts.Cancel();
            _cts = new CancellationTokenSource();
            Kontakte.Clear();
            _lädt = true;
            using FbController2 fbController = new FbController2();
            Filter.Seite = seite;
            var token = _cts.Token;

            await foreach (var kontakt in lieferantenkontaktService.GetAsync(Filter, fbController, token))
            {
                Kontakte.Add(kontakt);
                StateHasChanged();
            }

            TotalItems = await lieferantenkontaktService.GetTotalAmountAsync(Filter, fbController, token);


            _lädt = false;
        }

#nullable disable
        [CascadingParameter] public AppMitarbeiter Mitarbeiter { get; set; }
#nullable enable

        private async Task DownloadLieferantenKontakt(LieferantenKontakt input, bool mitBriefpapier = false)
        {
            BriefView view = new BriefView(input.LFKO_A_BETREFF, input.LFKO_B_LANGNOTIZ ?? String.Empty);

            Lieferant? lieferant = await Lieferant.GetLieferantAsync(input.LFKO_N_LFNR);
            LieferantenPartner? partner = await LieferantenPartner.GetLieferantenPartnerAsync(input.LFKO_N_LFNR, input.LFKO_N_PARTNER);

            if (partner is not null)
            {
                await view.SetEmpfängerAsync(partner);
            }
            else
            {
                if (lieferant is not null)
                {
                    await view.SetEmpfängerAsync(lieferant);
                }
            }

            Personal? versender = await Personal.GetPersonalAsync(Mitarbeiter.PersonalNummer);
            if (versender is not null)
            {
                view.SetAbsender(versender);
            }

            BriefRegelsatz briefRegeln = new BriefRegelsatz()
            {
                ShowFooter = mitBriefpapier,
                ShowHeader = mitBriefpapier
            };

            PrintBrief printBrief = await PrintBrief.CreateAsync(view, briefRegeln);

            byte[] file = printBrief.PrintToMemory();

            await downloadService.DownloadFile($"Brief-{input.LFKO_N_LFNR}.pdf", file, "application/pdf");
            await downloadService.ClearBuffers();
        }

        private async Task SaveAsync(LieferantenKontakt input)
        {
            Kontakte.Insert(0, input);
            _selectedLieferantenKontakt = null;
            await jsRuntime.ShowToast(ToastType.success, "Lieferantenkontakt wurde angelegt.");
        }

        private async Task EditAsync(LieferantenKontakt input)
        {
            _selectedLieferantenKontakt = null;
            await jsRuntime.ShowToast(ToastType.success, "Lieferantenkontakt wurde geändert.");
        }

        private async Task DeleteAsync(LieferantenKontakt input)
        {
            Kontakte.Remove(input);
            _selectedLieferantenKontakt = null;
            await jsRuntime.ShowToast(ToastType.success, "Lieferantenkontakt wurde gelöscht.");
        }


        public Task AddAsync()
        {
            DateTime currentDateTime = DateTime.Now;
            DateTime wiedervorlageDateTime = currentDateTime.AddDays(7);

            _selectedLieferantenKontakt = new LieferantenKontakt
            {
                LFKO_N_LFNR = LieferantenId,
                LFKO_N_WIEDERV_USER = Mitarbeiter.PersonalNummer,
                LFKO_D_DATE = currentDateTime,
                LFKO_T_TIME = currentDateTime,
                LFKO_D_WIEDERV_DATE = wiedervorlageDateTime,
                LFKO_T_WIEDERV_TIME = wiedervorlageDateTime,
                LFKO_N_BELEGNUMMER = Belegnummer,
                LFKO_A_BELEGTYP = Belegtyp
            };

            StateHasChanged();
            return Task.CompletedTask;
        }
        private Task CancelAsync()
        {
            _selectedLieferantenKontakt = null;
            return Task.CompletedTask;
        }

       

        protected override async Task OnParametersSetAsync()
        {
            Filter.LieferantenId = LieferantenId;
            await LadeAsync(1);
        }

        private string GetDisplayClass(LieferantenKontakt kontakt)
        {
            if (kontakt.LFKO_L_WIEDERV_ERL)
            {
                return "table-success";
            }

            return string.Empty;
        }


        public ValueTask DisposeAsync()
        {
            _cts?.Cancel();
            _cts?.Dispose();

            return ValueTask.CompletedTask;
        }
    }
}
