﻿using Microsoft.AspNetCore.Components;
using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using WK5.Core;
using WK5.Core.Basis.Filter;
using WK5.Core.Models;

namespace WK5_Blazor.Components
{
    public partial class Kundenbelege : IAsyncDisposable
    {


        [Parameter] public KundenBelegeFilter Filter { get; set; } = new KundenBelegeFilter();
        public int TotalItems { get; set; }
        private int _seite = 1;
        private bool _showSucheArtikelModal;
        private bool _isDisposed;
        public int Seite
        {
            get => _seite;
            set
            {
                if (value <= 0)
                {
                    value = 1;
                }

                _seite = value;
            }
        }

        private async Task ArtikelSelectedAsync(Artikel artikel)
        {
            Filter.Artikelnummer = artikel.Artikelnummer;
            _showSucheArtikelModal = false;
            await Suchen();
        }

        public bool IsLoading { get; set; } = false;
        private bool _isLoadingBeleg;
        private CancellationTokenSource _cts  = new CancellationTokenSource();
        public List<KundenBeleg> Ergebnis { get; set; } = new List<KundenBeleg>();

        protected override async Task OnParametersSetAsync()
        {
            if (!_isDisposed)
            {
                await Suchen();
            }
        }

        private async Task Suchen()
        {
            IsLoading = true;
            StateHasChanged();
            TotalItems = await belegService.GetAnzahlKundenbelegeAsync(Filter);

            Filter.Seite = Seite;
            _cts.Cancel();
            _cts.Dispose();
            _cts = new CancellationTokenSource();
            Ergebnis.Clear();
            var token = _cts.Token;

            try
            {
                await foreach (var result in belegService.SucheKundenbelegeAsync(Filter, token))
                {
                    Ergebnis.Add(result);
                    StateHasChanged();
                }
            }
            catch (OperationCanceledException){}
            



            IsLoading = false;
        }

        private async Task SeiteChanged(int page)
        {
            Seite = page;
            await Suchen();
        }

        private KundenBeleg? SelectedBeleg { get; set; }
        private Beleg? _beleg;

        private Dictionary<BelegTyp, List<int>> _subBelege = new Dictionary<BelegTyp, List<int>>();
        private async Task SelectBelegAsync(KundenBeleg beleg)
        {

            _isLoadingBeleg = true;
            if (SelectedBeleg == beleg)
            {
                SelectedBeleg = null;
                _beleg = null;
                _subBelege.Clear();
            }
            else
            {
                SelectedBeleg = beleg;

                if (beleg.Belegtyp.Equals("RM", System.StringComparison.OrdinalIgnoreCase))
                {
                    _beleg = null;
                    _subBelege.Clear();
                }
                else
                {

                    using FbController2 fbController = new FbController2();

                    // Wir greifen hier nicht auf die Instanz zu, da es bei einem Doppelklick sein kann, dass SelectedBeleg auf null gestezt wurde
                    _beleg = EnumHelper.GetBelegTyp(beleg.Belegtyp) switch
                    {
                        BelegTyp.Angebot => await Angebot.GetAngebotAsync(beleg.Belegnummer, fbController),
                        BelegTyp.Auftrag => await Auftrag.GetAuftragAsync(beleg.Belegnummer, fbController),
                        BelegTyp.Lieferschein => await Lieferschein.GetLieferscheinAsync(beleg.Belegnummer, fbController),
                        BelegTyp.Rechnung => await Rechnung.GetRechnungAsync(beleg.Belegnummer, fbController),
                        BelegTyp.Gutschrift => await Gutschrift.GetGutschriftAsync(beleg.Belegnummer, fbController),
                        _ => null
                    };


                    _subBelege = await belegService.GetSubBelege(EnumHelper.GetBelegTyp(beleg.Belegtyp), beleg.Belegnummer, fbController);
                }
            }

            _isLoadingBeleg = false;

        }

        public ValueTask DisposeAsync()
        {
            _cts?.Cancel();
            _cts?.Dispose();
            _isDisposed = true;

            return ValueTask.CompletedTask;
        }
    }
}
