﻿const Default = {
    "lineWidth": 1,
    "strokeStyle": "#222222"
};

const sigpadMap = new Map();

export default class Sigpad {


    static getOrCreateInstance(element, config) {
        return this.getInstance(element) || new this(element, typeof config === 'object' ? config : null)
    }

    static getInstance(element) {
        if (sigpadMap.has(element)) {
            return sigpadMap.get(element);
        }
        return null;
    }

    static getAllInstances() {
        var instances = [];
        sigpadMap.forEach((value, key, map) => {
            instances.push(value);
        });
        return instances;
    }

    constructor(element, options) {
        if (!sigpadMap.has(element)) {
            sigpadMap.set(element, this);
        }

        this._config = this._getConfig(options);
        this._element = element;
        this._mousePos = null;
        this._lastPos = null;
        this._drawing = false;

        this._setListeners();
        this._applyOptions();


        this._element.width = this._element.clientWidth;
    }

    destroy() {
        if (!sigpadMap.has(this._element)) {
            return
        }

        sigpadMap.delete(this._element);

        this._element.removeEventListener("mousedown", this._onStart);
        this._element.removeEventListener("mouseup", this._onEnd);
        this._element.removeEventListener("mousemove", this._onMove);

        this._element.removeEventListener("touchstart", this._onStart);
        this._element.removeEventListener("touchend", this._onEnd);
        this._element.removeEventListener("touchmove", this._onMove);
    }

    _applyOptions(ctx = null) {
        if (ctx == null) {
            ctx = this._element.getContext("2d");
        }
        ctx.strokeStyle = this._config.strokeStyle;
        ctx.lineWidth = this._config.lineWidth;
    }

    _getRelativeMousePos(e) {
        var r = this._element.getBoundingClientRect();

        var x = e.clientX;
        var y = e.clientY;

        var startX = r.left;
        var startY = r.top;
        var w = r.right - r.left;
        var h = r.bottom - r.top;

        var endX = startX + w;
        var endY = startY + h;

        if (x < startX || y < startY || x > endX || y > endY) {
            return null;
        } else {
            return {
                x: x - startX,
                y: y - startY
            };
        }
    }

    clear() {
        var ctx = this._element.getContext("2d");
        ctx.clearRect(0, 0, this._element.width, this._element.height);
        ctx.save();

        this._mousePos = null;
        this._lastPos = null;
        this._drawing = false;
    }

    getImage() {
        return this._element.toDataURL();
    }

    setImage(image) {
        var ctx = this._element.getContext("2d");
        var img = new Image();
        img.src = image;
        img.onload = function () {
            ctx.drawImage(img, 0, 0);
        }
    }

    _getConfig(config) {
        return {
            ...Default,
            ...config,
            ...(typeof config === 'object' && config ? config : {})
        }
    }


    _setListeners() {
        this._element.addEventListener("mousedown", this._onStart);
        this._element.addEventListener("mouseup", this._onEnd);
        this._element.addEventListener("mousemove", this._onMove);

        this._element.addEventListener("touchstart", this._onStart);
        this._element.addEventListener("touchend", this._onEnd);
        this._element.addEventListener("touchmove", this._onMove);
    }

    _onStart(mouseEvent) {
        if ("touches" in mouseEvent) {
            mouseEvent.preventDefault();
            mouseEvent = mouseEvent.touches[0];
        }

        var data = Sigpad.getInstance(this);

        var mousePos = data._getRelativeMousePos(mouseEvent);
        if (mousePos != null) {
            data._mousePos = null;
            data._lastPos = null;
            data._lastPos = mousePos;
            data._drawing = true;

            const event = new CustomEvent('sigpad.start', { detail: data.getImage() });
            data._element.dispatchEvent(event);
        }
    }

    _onEnd(mouseEvent) {
        if ("touches" in mouseEvent) {
            mouseEvent.preventDefault();
            mouseEvent = mouseEvent.touches[0];
        }

        var data = Sigpad.getInstance(this);

        data._drawing = false;

        const event = new CustomEvent('sigpad.finish', { detail: data.getImage() });
        data._element.dispatchEvent(event);
    }

    _onMove(mouseEvent) {
        if ("touches" in mouseEvent) {
            mouseEvent.preventDefault();
            mouseEvent = mouseEvent.touches[0];
        }

        var data = Sigpad.getInstance(this);

        if (data._drawing) {

            var pos = data._getRelativeMousePos(mouseEvent);            
            if (pos != null) {
                data._mousePos = pos;
                const event = new CustomEvent('sigpad.move', { detail: data.getImage() });
                data._element.dispatchEvent(event);
            } else {
                data._onEnd();
            }
        }
    }

    render() {
        if (this._element.width != this._element.clientWidth) {
            this._element.width = this._element.clientWidth;
        }
        if (this._drawing && (this._lastPos != null && typeof this._lastPos != "undefined") && (this._mousePos != null && typeof this._mousePos != "undefined")) {

            var ctx = this._element.getContext("2d");
            ctx.beginPath();
            this._applyOptions(ctx);
            ctx.moveTo(this._lastPos.x, this._lastPos.y);
            ctx.lineTo(this._mousePos.x, this._mousePos.y);
            ctx.closePath();
            ctx.stroke();
            this._lastPos = this._mousePos;
        }
    }
}



const requestAnimFrame = (function (callback) {
    return window.requestAnimationFrame ||
        window.webkitRequestAnimationFrame ||
        window.mozRequestAnimationFrame ||
        window.oRequestAnimationFrame ||
        window.msRequestAnimaitonFrame ||
        function (callback) {
            window.setTimeout(callback, 1000 / 144);
        };
})();

(function renderSignatures() {
    requestAnimFrame(renderSignatures);
    Sigpad.getAllInstances().forEach((sigpad, index) => {
        sigpad.render();
    });

})();
