using Messeportal.Core.Services;
using Microsoft.AspNetCore.Authentication.Cookies;
using Microsoft.AspNetCore.Authentication;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.RazorPages;
using System.ComponentModel.DataAnnotations;
using System.Net;
using System.Security.Claims;
using System.Xml.Linq;
using Messeportal.Core.Models;
using ScholzLibrary;
using System.DirectoryServices.Protocols;
using Microsoft.AspNetCore.Identity;
using ScholzLibrary.Basis.Erweiterungen;

namespace Messeportal.Pages.Account
{
    [AllowAnonymous]
    public class LoginModel : PageModel
    {
        [BindProperty]
        public LoginInput Input { get; set; } = new LoginInput();
        public string? ReturnUrl { get; set; }

        public async Task OnGetAsync(string? returnUrl = null)
        {
            ReturnUrl = returnUrl ?? Url.Content("~/");

            try
            {
                // Clear the existing external cookie
                await HttpContext.SignOutAsync(CookieAuthenticationDefaults.AuthenticationScheme);
            }
            catch { }

        }


        public async Task<IActionResult> OnPostAsync(string? returnUrl = null)
        {
            returnUrl = returnUrl ?? Url.Content("~/");

            if (ModelState.IsValid)
            {

                // Erst prfen wir gegen die Datenbank
                MitarbeiterService mitarbeiterService = new MitarbeiterService();
                using SqlController sqlController = new SqlController(AppdatenService.ConnectionString);
                Mitarbeiter? mitarbeiter = await mitarbeiterService.GetAsync(Input.Username, sqlController);

                // Lokale Konten mssen als ersten geprft werden.
                if (mitarbeiter is not null)
                {
                    PasswordHasher<Mitarbeiter> hasher = new PasswordHasher<Mitarbeiter>();

                    string passwordHashed = hasher.HashPassword(mitarbeiter, Input.Password + mitarbeiter.Salt);

                    PasswordVerificationResult result = hasher.VerifyHashedPassword(mitarbeiter, mitarbeiter.Passwort, Input.Password + mitarbeiter.Salt);
                    // Das Handling luft spter auf Basis des Objektes ab.
                    if (result is PasswordVerificationResult.Failed)
                    {
                        mitarbeiter = null;
                    }
                }
                else
                {
                    // Wenn kein lokales Konto gefunden wurde, dann prfen wir das Active-Directory

                    try
                    {
                        using var connection = new LdapConnection(AppdatenService.LdapServer);

                        var networkCredential = new NetworkCredential(Input.Username, Input.Password, AppdatenService.LdapDomainServer);
                        connection.SessionOptions.SecureSocketLayer = false; // Warnung kann ignoriert werden, ist ein Fehler vom Package
                        connection.AuthType = AuthType.Negotiate;
                        connection.Bind(networkCredential);

                        var searchRequest = new SearchRequest
                            (
                            AppdatenService.LdapDistinguishedName,
            $"(SAMAccountName={Input.Username})",
                            SearchScope.Subtree, new string[]
                            {
                                "cn",
                                "mail",
                                "givenName",
                                "sn",
                                "objectGUID"
                            });

                        SearchResponse directoryResponse = (SearchResponse)connection.SendRequest(searchRequest);

                        SearchResultEntry searchResultEntry = directoryResponse.Entries[0];

                        Dictionary<string, string> attributes = new Dictionary<string, string>();
                        Guid? guid = null;
                        foreach (DirectoryAttribute userReturnAttribute in searchResultEntry.Attributes.Values)
                        {
                            if (userReturnAttribute.Name == "objectGUID")
                            {
                                byte[] guidByteArray = (byte[])userReturnAttribute.GetValues(typeof(byte[]))[0];
                                guid = new Guid(guidByteArray);
                                attributes.Add("guid", ((Guid)guid).ToString());
                            }
                            else
                            {
                                attributes.Add(userReturnAttribute.Name, (string)userReturnAttribute.GetValues(typeof(string))[0]);
                            }
                        }

                        if (!attributes.ContainsKey("mail"))
                        {
                            attributes.Add("mail", string.Empty);
                        }

                        if (!attributes.ContainsKey("sn"))
                        {
                            attributes.Add("sn", string.Empty);
                        }

                        if (!attributes.ContainsKey("givenName"))
                        {
                            attributes.Add("givenName", string.Empty);
                        }

                        if (guid is null)
                        {
                            throw new InvalidOperationException();
                        }

                        mitarbeiter = await mitarbeiterService.GetAsync((Guid)guid, sqlController);

                        if (mitarbeiter is null)
                        {
                            mitarbeiter = new Mitarbeiter
                            {
                                Username = Input.Username.ToUpper(),
                                Guid = (Guid)guid,
                                Email = attributes["mail"],
                                Anzeigename = $"{attributes["givenName"]} {attributes["sn"]}",
                                Herkunft = "ad"
                            };

                            // Wenn der erste User angelegt wird, dann sollen diesem alle Berechtigungen gegeben werden
                            if (!MitarbeiterService.ActiveDirectoryUserExists)
                            {
                                foreach (var berechtigung in AppdatenService.Berechtigungen)
                                {
                                    mitarbeiter.Berechtigungen.Add(berechtigung);
                                }
                            }

                            await mitarbeiterService.CreateAsync(mitarbeiter, sqlController);
                            await mitarbeiterService.UpdateBerechtigungenAsync(mitarbeiter, sqlController);


                            MitarbeiterService.ActiveDirectoryUserExists = true;
                        }



                    }
                    catch (LdapException ex)
                    {

                    }
                }

                // Wenn wir ein Mitarbeiter Objekt haben, dann knnen wir uns einloggen, ansonsten ist irgendwas schief gelaufen
                if (mitarbeiter is not null)
                {
                    var claims = new List<Claim>
                    {
                        new Claim("userId", mitarbeiter.UserId.ToString()),
                    };

                    foreach (var berechtigung in mitarbeiter.Berechtigungen)
                    {
                        claims.Add(new Claim(ClaimTypes.Role, berechtigung.Identifier));
                    }

                    var claimsIdentity = new ClaimsIdentity(claims, CookieAuthenticationDefaults.AuthenticationScheme);

                    var authProperties = new AuthenticationProperties
                    {
                        IsPersistent = Input.RememberMe,
                        RedirectUri = returnUrl
                    };

                    await HttpContext.SignInAsync(CookieAuthenticationDefaults.AuthenticationScheme, new ClaimsPrincipal(claimsIdentity), authProperties);
                    return LocalRedirect(returnUrl);
                }
                else
                {
                    ModelState.AddModelError("login-error", "Username oder Passwort ist falsch.");
                }
            }
            return Page();
        }
    }

    public class LoginInput
    {
        [Required]
        public string Username { get; set; } = String.Empty;
        [Required]
        [DataType(DataType.Password)]
        public string Password { get; set; } = String.Empty;
        [Display(Name = "Eingeloggt bleiben")]
        public bool RememberMe { get; set; }
    }
}
