﻿using Messeportal.Core.Filters;
using Messeportal.Core.Models;
using ScholzLibrary;
using System.Text;

namespace Messeportal.Core.Services
{
    public class MitarbeiterService : IModelService<Mitarbeiter, int, BenutzerFilter>
    {
        public static bool ActiveDirectoryUserExists { get; set; }
        public async Task CreateAsync(Mitarbeiter input, SqlController sqlController)
        {
            string sql = @"INSERT INTO users
(
username,
anzeigename,
active_directory_guid,
email,
passwort,
salt,
herkunft
)
VALUES 
(
@USERNAME,
@ANZEIGENAME,
@ACTIVE_DIRECTORY_GUID,
@EMAIL,
@PASSWORT,
@SALT,
@HERKUNFT
) SELECT SCOPE_IDENTITY();";

            input.UserId = await sqlController.GetFirstAsync<int>(sql, new
            {
                USERNAME = input.Username,
                ANZEIGENAME = input.Anzeigename,
                ACTIVE_DIRECTORY_GUID = input.Guid,
                EMAIL = input.Email,
                PASSWORT = input.Passwort,
                SALT = input.Salt,
                HERKUNFT = input.Herkunft
            });
        }

        public async Task<Mitarbeiter?> GetAsync(int userId, SqlController sqlController)
        {
            string sql = @"SELECT * FROM users WHERE user_id = @USER_ID";

            var user = await sqlController.GetFirstAsync<Mitarbeiter>(sql, new
            {
                USER_ID = userId
            });

            if (user is not null)
            {
                user.Berechtigungen = await GetUserBerechtigungAsync(user, sqlController);
            }

            return user;
        }

        public async Task<Mitarbeiter?> GetAsync(Guid guid, SqlController sqlController)
        {
            string sql = @"SELECT * FROM users WHERE active_directory_guid = @ACTIVE_DIRECTORY_GUID AND herkunft = 'ad'";

            var user = await sqlController.GetFirstAsync<Mitarbeiter>(sql, new
            {
                ACTIVE_DIRECTORY_GUID = guid
            });

            if (user is not null)
            {
                user.Berechtigungen = await GetUserBerechtigungAsync(user, sqlController);
            }
            return user;
        }
        public async Task<Mitarbeiter?> GetAsync(string username, SqlController sqlController)
        {
            string sql = @"SELECT * FROM users WHERE UPPER(username) = UPPER(@USERNAME) AND herkunft = 'lokal'";

            var user = await sqlController.GetFirstAsync<Mitarbeiter>(sql, new
            {
                USERNAME = username
            });

            if (user is not null)
            {
                user.Berechtigungen = await GetUserBerechtigungAsync(user, sqlController);
            }

            return user;
        }

        private async Task<List<Berechtigung>> GetUserBerechtigungAsync(Mitarbeiter mitarbeiter, SqlController sqlController)
        {
            var list = await sqlController.SelectDataAsync<Berechtigung>(@"SELECT b.*
FROM user_berechtigungen ub
INNER JOIN berechtigungen b ON (b.berechtigung_id = ub.berechtigung_id)
WHERE user_id = @USER_ID
", new
            {
                USER_ID = mitarbeiter.UserId
            });

            return list;
        }
        public async Task<List<Mitarbeiter>> GetAsync(BenutzerFilter filter, SqlController sqlController)
        {
            StringBuilder sqlBuilder = new();
            sqlBuilder.Append("SELECT * FROM users WHERE 1 = 1");
            sqlBuilder.AppendLine(GetFilterWhere(filter));
            sqlBuilder.AppendLine(@$"  ORDER BY user_id DESC");
            sqlBuilder.AppendLine($" OFFSET {(filter.Seite - 1) * filter.Limit} ROWS FETCH NEXT {filter.Limit} ROWS ONLY");

            // Zum Debuggen schreiben wir den Wert einmal als Variabel
            string sql = sqlBuilder.ToString();

            List<Mitarbeiter> list = await sqlController.SelectDataAsync<Mitarbeiter>(sql, GetFilterParameter(filter));

            return list;
        }

        public object? GetFilterParameter(BenutzerFilter filter)
        {
            return new
            {
                SUCHBEGRIFF = $"%{filter.Suchbegriff}%"
            };
        }

        public string GetFilterWhere(BenutzerFilter filter)
        {
            StringBuilder sb = new StringBuilder();

            if (!string.IsNullOrWhiteSpace(filter.Suchbegriff))
            {
                sb.AppendLine(@" AND 
(
        UPPER(anzeigename) LIKE @SUCHBEGRIFF
    OR  UPPER(email) LIKE @SUCHBEGRIFF
    OR  UPPER(username) LIKE @SUCHBEGRIFF
)");
            }

            string sql = sb.ToString();
            return sql;
        }

        public async Task<int> GetTotalAsync(BenutzerFilter filter, SqlController sqlController)
        {
            string sql = "SELECT COUNT(*) FROM users WHERE 1 = 1";
            sql += GetFilterWhere(filter);

            int result = await sqlController.GetFirstAsync<int>(sql, GetFilterParameter(filter));

            return result;
        }

        public async Task UpdateAsync(Mitarbeiter input, SqlController sqlController)
        {
            string sql = @"UPDATE users SET
username = @USERNAME,
anzeigename = @ANZEIGENAME,
email = @EMAIL,
WHERE user_id = @USER_ID";

            await sqlController.QueryAsync(sql, new
            {
                USERNAME = input.Username,
                ANZEIGENAME = input.Anzeigename,
                EMAIL = input.Email,
                USER_ID = input.UserId
            });
        }

        public async Task UpdateBerechtigungenAsync(Mitarbeiter input, SqlController sqlController)
        {
            // User Berechtigungen aktualisieren
            // Erst alles entfernen
            string sql = "DELETE FROM user_berechtigungen WHERE user_id = @USER_ID";
            await sqlController.QueryAsync(sql, new
            {
                USER_ID = input.UserId
            });

            // Berechtigungen einfügen
            foreach (var berechtigung in input.Berechtigungen)
            {
                sql = @"INSERT INTO user_berechtigungen
(
user_id,
berechtigung_id
)
VALUES
(
@USER_ID,
@BERECHTIGUNG_ID
)";

                await sqlController.QueryAsync(sql, new
                {
                    USER_ID = input.UserId,
                    BERECHTIGUNG_ID = berechtigung.BerechtigungId
                });

            }
        }

        public async Task DeleteAsync(Mitarbeiter input, SqlController sqlController)
        {
            string sql = "DELETE FROM users WHERE user_id = @USER_ID";

            await sqlController.QueryAsync(sql, new
            {
                USER_ID = input.UserId,
            });

        }
    }
}
